﻿using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.NoticeOfConnection.DocumentConsumption
{
    /// <summary>
    /// Conformance Test Specification: PCEHR B2B Gateway Notice of Connection Test Plan
    ///                                 Version 4.0 — 4 March 2013
    /// Operation:  getDocument
    /// </summary>
    [TestClass]
    public class NOC_GetDocument : CcaTest
    {
        /// <summary>
        /// Test No:         42
        /// Objective/Input:
        /// To ensure that the client system can execute a retrieve document
        /// operation successfully.
        ///
        /// Expected Result:
        /// The request should be transmitted successfully to the PCEHR System.
        /// Upon successful execution, the requested document is received from
        /// the PCEHR system with the ResponseStatusType: SUCCESS in the PCEHR
        /// Header.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_042()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 13");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            var accessResponse = ProxyHelper.PcehrProxy.GainAccessWithOutCode(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessWithoutCode);

            // If you need to choose a different document, uncomment this to get the list:
            ////var listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            ////LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);

            DocumentRequest request = new DocumentRequest();
            request.DocumentUniqueId = "1.2.36.1.2001.1005.26.1.1.4.3662^3662";
            request.RepositoryUniqueId = "1.2.36.1.2001.1007.10.8003640002000050";
            request.SaveDocument = true;
            DocumentResponse pcehrResponse = ProxyHelper.PcehrProxy.RetrieveDocument(identifier, user, request);
            
            // PCEHR SVT Support has asked to see OK response. If you see DemographicMismatchWarning
            // then choose a different document.
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, pcehrResponse.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
        }

        /// <summary>
        /// Test No:         43
        /// Objective/Input:
        /// To show that the client system can handle an XDSRepositoryError
        /// (PCEHR_ERROR_3503 - Removed document not retrievable from PCEHR)
        ///
        /// Expected Result:
        /// The request should be transmitted successfully to the PCEHR System.
        /// The client system is able to handle the error XDSRepositoryError
        /// (PCEHR_ERROR_3503 - Removed document not retrievable from PCEHR)
        /// and inform the user appropriately.
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_043()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            //DocumentQuery query = new DocumentQuery();
            //query.DocumentStatus = new List<DocumentStatus> {DocumentStatus.Deprecated};
            //DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentList(identifier, user, query);
            //LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListWithQuery);
            //LogAssert.AreNotEqual(0, listResponse.DocumentList.Count, DialogueResource.NumberOfDocumentsInList);

            DocumentRequest request = new DocumentRequest();

            //request.DocumentUniqueId = listResponse.DocumentList[0].DocumentUniqueId;
            request.DocumentUniqueId = "2.25.329060028460708112308198229900946655326";
            //request.RepositoryUniqueId = listResponse.DocumentList[0].RepositoryUniqueId;
            request.RepositoryUniqueId = "1.2.36.1.2001.1007.10.8003640002000050";
            request.SaveDocument = true;
            DocumentResponse pcehrResponse = ProxyHelper.PcehrProxy.RetrieveDocument(identifier, user, request);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, pcehrResponse.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
            LogAssert.AreEqual("PCEHR_ERROR_3503", pcehrResponse.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("Removed document not retrievable from PCEHR", pcehrResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
        }

        /// <summary>
        /// Test No:         44
        /// Objective/Input:
        /// Perform a DocumentRepository_RetrieveDocumentSet operation with a
        /// document id which does not exist in the PCEHR System. Set
        /// DocumentUniqueId to ID of document which does not exist in the
        /// PCEHR system.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The
        /// client system is able to handle the status response type Failure,
        /// the errorCode XDSRepositoryError, and the codeContext
        /// (%Actual Error Message%).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_044()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            DocumentRequest request = new DocumentRequest();
            request.DocumentUniqueId = "abcdef";
            request.RepositoryUniqueId = "1.2.36.1.2001.1007.10.8003640002000050";
            request.SaveDocument = true;
            DocumentResponse pcehrResponse = ProxyHelper.PcehrProxy.RetrieveDocument(identifier, user, request);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, pcehrResponse.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
            LogAssert.AreEqual("PCEHR_ERROR_3501", pcehrResponse.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("No metadata found", pcehrResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
        }

        /// <summary>
        /// Test No:         45
        /// Objective/Input:
        /// Perform a DocumentRepository_RetrieveDocumentSet operation with an
        /// IHI which does not match the patient id in the document.
        /// IHI number in header is different to the IHI of the requested document.
        /// Document Unique Id set to document of different patient.
        ///
        /// Expected Result:
        /// The request is transmitted successfully to the PCEHR System. The
        /// client system is able to handle the response details status
        /// Response Type Failure, XDSRepositoryError, and codeContext
        /// (%Actual Error Message%).
        /// </summary>
        [TestMethod]
        [TestCategory("PCEHR_NOC")]
        public void PCEHR_NOC_045()
        {
            // Input data from "PCEHR NOC Test Data NEHTA HIPS V4.4", NOC Test Data Master, Test 45, IHI column.
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "NOC ID 1");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;

            DocumentRequest request = new DocumentRequest();

            // Input data from "PCEHR NOC Test Data NEHTA HIPS V4.4", NOC Test Data Master, Test 45, Document ID column.
            request.DocumentUniqueId = "1.2.13.1.3998.468";

            // Input data from "PCEHR NOC Test Data NEHTA HIPS V4.4", NOC Test Data Master, Test 45, Repository ID column.
            request.RepositoryUniqueId = "1.2.36.1.2001.1007.10.8003640002000050";
            request.SaveDocument = true;
            DocumentResponse pcehrResponse = ProxyHelper.PcehrProxy.RetrieveDocument(identifier, user, request);
            LogAssert.ExpectResponse(HipsResponseIndicator.PcehrServiceError, pcehrResponse.HipsResponse, DialogueResource.HipsServiceRetrieveDocument);
            LogAssert.AreEqual("PCEHR_ERROR_3002", pcehrResponse.HipsResponse.ResponseCode, DialogueResource.PcehrResponseCode);
            LogAssert.AreEqual("Document metadata failed validation", pcehrResponse.HipsResponse.ResponseCodeDescription, DialogueResource.PcehrResponseDescription);
        }
    }
}