﻿using System;
using HIPS.AppServer.HIPSServiceHost.Services;
using HIPS.Common.DataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.HL7;
using HIPS.HibIntegration.Loader;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.ServiceInterfaces;

namespace HIPS.AppServer.ServiceHost
{
    /// <summary>
    /// This class implements web services for processing HL7 messages from the Enterprise Service Bus / Health Integration Broker.
    /// </summary>
    public class DatabaseLoaderService : BaseService, IDatabaseLoaderService
    {
        // Useful for debugging to check that the message was all parsed properly
        private readonly bool IncludeOriginalMessageInReply = false;

        /// <summary>
        /// Notifies the pas event.
        /// </summary>
        /// <param name="messageForm">The message form.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        public string NotifyPasEvent(string messageForm, UserDetails user)
        {
            string reply;
            MessageLoader loader = new MessageLoader();
            HL7Message parsedMessage;
            try
            {
                parsedMessage = HL7Message.Parse(messageForm);
            }
            catch (HL7ParseException ex)
            {
                string errorMessage = string.Format(ResponseStrings.UnparsableMessage, messageForm);
                EventLogger.WriteLog(errorMessage, ex, user, LogMessage.HIPS_MESSAGE_114);
                return HL7AcknowledgementFactory.AcknowledgeUnparseableMessage(ex.Message).Encode();
            }
            catch (Exception ex)
            {
                string errorMessage = string.Format(ResponseStrings.UnparsableMessage, messageForm);
                EventLogger.WriteLog(errorMessage, ex, user, LogMessage.HIPS_MESSAGE_115);
                return HL7AcknowledgementFactory.AcknowledgeUnparseableMessage(errorMessage).Encode();
            }
            HL7MessageLog testLog;
            HL7Message ack = loader.Process(parsedMessage, messageForm, user, out testLog);
            if (IncludeOriginalMessageInReply)
            {
                reply = parsedMessage.Encode() + "\r\r" + ack.Encode();
            }
            else
            {
                reply = ack.Encode();
            }
            return reply;
        }

        /// <summary>
        /// Notifies the path event.
        /// </summary>
        /// <param name="messageForm">The message form.</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        public string NotifyPathEvent(string messageForm, UserDetails user)
        {
            string reply;
            MessageLoader loader = new MessageLoader();
            HL7MessageLog testLog;
            HL7Message message = HL7Message.Parse(messageForm);
            HL7Message ack = loader.Process(message, messageForm, user, out testLog);
            if (IncludeOriginalMessageInReply)
            {
                reply = message.Encode() + "\r\r" + ack.Encode();
            }
            else
            {
                reply = ack.Encode();
            }
            return reply;
        }
    }
}