﻿using System;
using System.Linq;
using System.Text.RegularExpressions;
using System.Xml;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.CommonBusinessLogic.Cda
{
    /// <summary>
    /// Validation logic for CDA documents.
    /// </summary>
    public class CdaValidation
    {
        /// <summary>
        /// OID prefix for healthcare identifiers.
        /// </summary>
        private const string HI_OID_PREFIX = "1.2.36.1.2001.1003.0.";

        /// <summary>
        /// Regular expression that matches a 16-digit number.
        /// </summary>
        private const string SIXTEEN_DIGIT_REGEX = "^\\d{16}$";
        
        /// <summary>
        /// Validation parameters for an HPI-I.
        /// </summary>
        private readonly HiValidationParameters hpiiParams = new HiValidationParameters() 
        {
            FirstSixDigits = "800361", 
            Description = "HPI-I", 
            Type = HiType.HpiI
        };

        /// <summary>
        /// Validation parameters for an HPI-O.
        /// </summary>
        private readonly HiValidationParameters hpioParams = new HiValidationParameters() 
        { 
            FirstSixDigits = "800362", 
            Description = "HPI-O",
            Type = HiType.HpiO 
        };

        /// <summary>
        /// Validation parameters for an IHI.
        /// </summary>
        private readonly HiValidationParameters ihiParams = new HiValidationParameters() 
        {
            FirstSixDigits = "800360",
            Description = "IHI",
            Type = HiType.Ihi 
        };

        /// <summary>
        /// Synchronisation object to ensure that the fields in this object are only used by one thread at a time.
        /// </summary>
        private readonly object syncRoot = new object();

        /// <summary>
        /// The document type for the CDA document.
        /// </summary>
        private DocumentType documentType;

        /// <summary>
        /// The healthcare facility that is attempting to upload the CDA document to the PCEHR system.
        /// </summary>
        private Hospital hospital;

        /// <summary>
        /// The patient record that is the subject of care for the CDA document.
        /// </summary>
        private PatientMaster patient;

        /// <summary>
        /// The CDA document as an XML document.
        /// </summary>
        private XmlDocument xmlDoc;

        /// <summary>
        /// An XML namespace manager with "x" (HL7v3) and "ext" (Australian CDA extensions) entries.
        /// </summary>
        private XmlNamespaceManager xnm;

        /// <summary>
        /// Validates the Facility, Custodian and Author's HPI-O, the Author's HPI-I,
        /// the Patient's IHI and Date of Birth, and the Creation Time in the CDA document.
        /// </summary>
        /// <param name="xmlDoc">The CDA document</param>
        /// <param name="xnm">An XML namespace manager with "x" (HL7v3) and "ext" (Australian CDA extensions) entries.</param>
        /// <param name="documentType">The document's type.</param>
        /// <param name="hospital">The hospital whose HPI-O should match the HPI-O(s) in the document</param>
        /// <param name="patient">The patient whose IHI should match the IHI in the document</param>
        public void Validate(XmlDocument xmlDoc, XmlNamespaceManager xnm, DocumentType documentType, Hospital hospital, PatientMaster patient)
        {
            lock (this.syncRoot)
            {
                this.xmlDoc = xmlDoc;
                this.xnm = xnm;
                this.documentType = documentType;
                this.hospital = hospital;
                this.patient = patient;
                this.ValidateHpiOs();
                this.ValidateHpiIs();
                this.ValidateIhi();
                this.ValidateDateOfBirth();
                this.ValidateCreationTime();
            }
        }

        /// <summary>
        /// Validates the creation time in the CDA document.
        /// </summary>
        private void ValidateCreationTime()
        {
            string xpath = "/x:ClinicalDocument/x:effectiveTime";
            DateTime date = this.ValidateTimestamp(xpath, "Creation Time");
        }

        /// <summary>
        /// Validates the patient's date of birth in the CDA document.
        /// </summary>
        private void ValidateDateOfBirth()
        {
            string xpath = "/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/x:birthTime";

            DateTime date = this.ValidateTimestamp(xpath, "Patient DOB");
            if (date.Date != this.patient.RegisteredDateOfBirth.Date)
            {
                throw new Exception(string.Format(
                    "The patient DOB {0} in the document does not match the Registered DOB {1} in HIPS demographics.",
                    date.ToShortDateString(),
                    this.patient.RegisteredDateOfBirth.ToShortDateString()));
            }
        }

        /// <summary>
        /// Validates an "ext:id" node that should contain a national healthcare
        /// identifier (IHI, HPI-I or HPI-O). Throws an Exception if invalid.
        /// </summary>
        /// <param name="type">The parameters for validation of the particular
        ///   type of healthcare identifier</param>
        /// <param name="subject">A user-friendly description of which item within
        ///   the CDA document is identified by this identifier, e.g. "Facility"
        ///   or "Custodian" or "Author".</param>
        /// <param name="xpath">An XPath expression that matches the "ext:id" node
        ///   for the healthcare identifier</param>
        /// <param name="isMandatory">Whether this healthcare identifier is
        ///   mandatory. By default, it is considered mandatory.</param>
        /// <param name="allowAnyConfiguredHpio">If this is an HPI-O, whether to
        ///   allow any HPI-O configured in the HealthProviderOrganisation table. 
        ///   By default, the HPI-O must match that of the uploading hospital 
        ///   facility.</param>
        /// <returns>The value of the healthcare identifier, stripped of the OID
        /// prefix. Returns null if not found but not mandatory.</returns>
        private string ValidateHealthcareIdentifierNode(
            HiValidationParameters type, 
            string subject, 
            string xpath, 
            bool isMandatory = true,
            bool allowAnyConfiguredHpio = false)
        {
            XmlNode node = this.xmlDoc.SelectSingleNode(xpath, this.xnm);
            if (node == null && !isMandatory)
            {
                // It's not found, but it's not mandatory so ignore that.
                return null;
            }
            if (node == null || node.Attributes["root"] == null)
            {
                throw new Exception(string.Format("{0} {1} was not found in the CDA document at the path {2}.", subject, type.Description, xpath));
            }
            string value = node.Attributes["root"].Value;
            if (!value.StartsWith(HI_OID_PREFIX))
            {
                throw new Exception(string.Format("{0} {1} does not start with {2} and so is not a valid {1}.", subject, type.Description, HI_OID_PREFIX));
            }
            value = value.Substring(HI_OID_PREFIX.Length);
            if (!Regex.IsMatch(value, SIXTEEN_DIGIT_REGEX))
            {
                throw new Exception(string.Format("{0} {1} {2} does not consist of 16 digits and so is not a valid {1}.", subject, type.Description, value));
            }
            if (!value.StartsWith(type.FirstSixDigits))
            {
                throw new Exception(string.Format("{0} {1} {2} does not start with {3} and so is not a valid {1}.", subject, type.Description, value, type.FirstSixDigits));
            }
            if (!allowAnyConfiguredHpio && type.Type == HiType.HpiO && value != this.hospital.HpiO)
            {
                throw new Exception(string.Format(
                    "{0} {1} {2} in document does not match {1} {3} in HIPS configuration for hospital {4}",
                    subject, 
                    type.Description, 
                    value, 
                    this.hospital.HpiO, 
                    this.hospital.Description));
            }
            if (allowAnyConfiguredHpio && type.Type == HiType.HpiO)
            {
                var hpios = ListSingleton.Instance.AllHealthProviderOrganisations.Select(hpo => hpo.Hpio);
                if (!hpios.Contains(value))
                {
                    throw new Exception(string.Format(
                        "{0} HPI-O {1} in document does not match any HPI-O in HIPS configuration",
                        subject, 
                        value));
                }
            }
            return value;
        }

        /// <summary>
        /// Performs validation of HPI-I in a CDA document to meet NPDR NOC requirements.
        /// All types of document should contain the HPI-I in the Author section (author/assignedAuthor/assignedPerson),
        /// but it is not treated as mandatory here because of the HPI-I relaxation.
        /// </summary>
        private void ValidateHpiIs()
        {
            this.ValidateHealthcareIdentifierNode(
                this.hpiiParams, 
                "Author",
                "/x:ClinicalDocument/x:author/x:assignedAuthor/x:assignedPerson/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-I']",
                isMandatory: false);
        }

        /// <summary>
        /// <para>Performs additional validation of various HPI-O identifiers
        /// in a CDA document.
        /// </para>
        /// <para>All types of document must contain a Custodian section
        /// (custodian/assignedCustodian/representedCustodianOrganization),
        /// but the entity identifier for the custodian is not mandatory,
        /// and may not be an HPI-O. If it is an HPI-O, this method will
        /// enforce that the HPI-O is one of those configured in the
        /// HealthProviderOrganisation table, but it need not be the same
        /// HPI-O as the hospital that is uploading the document.
        /// </para>
        /// <para>Some types of document also contain the HPI-O in the Facility
        /// section (location/healthCareFacility/serviceProviderOrganisation)
        /// while other types contain the HPI-O in the Author section
        /// (author/assignedAuthor/assignedPerson/asEmployment/employerOrganisation).
        /// The facility and author HPI-O, if present, must match the hospital
        /// that is uploading the document.
        /// </para>
        /// </summary>
        private void ValidateHpiOs()
        {
            string[] documentTypesWithFacilityHpio =
            {
                DocumentTypeCodes.DischargeSummary,
                DocumentTypeCodes.PcehrPrescriptionRecord,
                DocumentTypeCodes.PcehrDispenseRecord
            };
            string[] documentTypesWithAuthorHpio =
            {
                DocumentTypeCodes.DischargeSummary,
                DocumentTypeCodes.EventSummary,
                DocumentTypeCodes.SpecialistLetter,
                DocumentTypeCodes.SharedHealthSummary
            };
            string documentTypeCode = this.documentType.Code;

            // Custodian: Represents the organization that is in charge of maintaining the document.
            // The custodian is the steward that is entrusted with the care of the document. Every
            // CDA document has exactly one custodian. The entity identifier of the custodian
            // organization is not mandatory, and any configured HPI-O is acceptable.
            this.ValidateHealthcareIdentifierNode(
                this.hpioParams,
                "Custodian",
                "/x:ClinicalDocument/x:custodian/x:assignedCustodian/x:representedCustodianOrganization/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-O']",
                isMandatory: false,
                allowAnyConfiguredHpio: true);

            // Facility: Details pertaining to the identification of a Healthcare
            // Organisation/Facility which is involved in or associated with the
            // delivery of the healthcare services to the patient, or caring for
            // his/her wellbeing. The value of one Entity Identifier SHALL be an
            // Australian HPI-O.
            if (documentTypesWithFacilityHpio.Contains(documentTypeCode))
            {
                this.ValidateHealthcareIdentifierNode(
                    this.hpioParams,
                    "Facility",
                    "/x:ClinicalDocument/x:componentOf/x:encompassingEncounter/x:location/x:healthCareFacility/x:serviceProviderOrganization/x:asOrganizationPartOf/x:wholeOrganization/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-O']");
            }

            // Author: The healthcare provider who is the main author of the document.
            // The author employer organisation is not mandatory.
            // The value of one Entity Identifier SHALL be an Australian HPI-O.
            if (documentTypesWithAuthorHpio.Contains(documentTypeCode))
            {
                this.ValidateHealthcareIdentifierNode(
                    this.hpioParams,
                    "Author",
                    "/x:ClinicalDocument/x:author/x:assignedAuthor/x:assignedPerson/ext:asEmployment/ext:employerOrganization/x:asOrganizationPartOf/x:wholeOrganization/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='HPI-O']",
                    isMandatory: false);
            }
        }

        /// <summary>
        /// Validates the entity identifier for the patient's IHI.
        /// </summary>
        private void ValidateIhi()
        {
            string xpath = "/x:ClinicalDocument/x:recordTarget/x:patientRole/x:patient/ext:asEntityIdentifier/ext:id[@assigningAuthorityName='IHI']";
            this.ValidateHealthcareIdentifierNode(this.ihiParams, "Patient", xpath);
        }

        /// <summary>
        /// Validates a timestamp in the CDA document. Throws an exception if not found, or not in a recognised format.
        /// </summary>
        /// <param name="xpath">An X-Path expression specifying the item's location within the CDA document.</param>
        /// <param name="itemName">The name of the item that should exist at the given path.</param>
        /// <returns>The date and time found at the given path.</returns>
        private DateTime ValidateTimestamp(string xpath, string itemName)
        {
            XmlNode node = this.xmlDoc.SelectSingleNode(xpath, this.xnm);
            if (node == null || node.Attributes["value"] == null)
            {
                throw new Exception(string.Format("{0} was not found in the document at the path {1}.", itemName, xpath));
            }
            string value = node.Attributes["value"].Value;
            DateTime? date = null;

            // Attempt to parse the timestamp.
            try
            {
                date = HL7DateTime.Parse(value);
            }
            catch (Exception)
            {
                throw new Exception(string.Format("The {0} {1} in the document does not have the expected format.", itemName, value));
            }

            if (!date.HasValue)
            {
                throw new Exception(string.Format("The {0} in the document was empty", itemName));
            }
            return date.Value;
        }
    }
}