﻿using System;
using System.Collections.Generic;
using System.Linq;

using AutoMapper;
using ModelSchemas = HIPS.CommonSchemas;
using NehtaSchemas = Nehta.VendorLibrary.CDA;

namespace HIPS.CommonBusinessLogic.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the custodian and facility entities.
    /// </summary>
    internal class FacilityProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            // Model --> NEHTA

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.IParticipationCustodian>()
              .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateCustodian())
              .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.ICustodian>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.Common.BaseCDAModel.CreateParticipantCustodian())
                .ForMember(dest => dest.Address, opt => opt.MapFrom(src => src.Hospital.Addresses.First()))
                .ForMember(dest => dest.ElectronicCommunicationDetail, opt => opt.MapFrom(src => src.Hospital.Contacts.First()))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.UniqueIdentifier, opt => opt.ResolveUsing((ModelSchemas.Cda.CdaDocumentDetails src) => Guid.NewGuid()));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.IOrganisationName>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.Common.BaseCDAModel.CreateOrganisationName())
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.Hospital.Name))
                .ForMember(
                    dest => dest.Identifiers,
                    opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
                    {
                        new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = "HPI-O",
                            AssigningGeographicArea = "National Identifier",
                            Root = "1.2.36.1.2001.1003.0." + src.Hospital.HpiO,
                        }
                    }));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.IParticipationFacility>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateFacility())
                .ForMember(dest => dest.Participant, opt => opt.MapFrom(src => src))
                .ForMember(dest => dest.Role, opt => opt.MapFrom(src => new ModelSchemas.Cda.CdaCode
                {
                    Code = "HOSP",
                    CodeSystem = "2.16.840.1.113883.1.11.17660",
                    CodeSystemName = "HL7 ServiceDeliveryLocationRoleType",
                    DisplayName = "Hospital"
                }));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.SCSModel.DischargeSummary.Interfaces.IFacility>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateParticipantForFacility())
                .ForMember(dest => dest.Addresses, opt => opt.MapFrom(src => src.Hospital.Addresses))
                .ForMember(dest => dest.ElectronicCommunicationDetails, opt => opt.MapFrom(src => src.Hospital.Contacts))
                .ForMember(dest => dest.Organisation, opt => opt.MapFrom(src => src));

            this.CreateMap<ModelSchemas.Cda.CdaDocumentDetails, NehtaSchemas.IOrganisation>()
                .ConstructUsing((ResolutionContext res) => NehtaSchemas.SCSModel.DischargeSummary.EDischargeSummary.CreateOrganisation())
                .ForMember(dest => dest.Department, opt => opt.MapFrom(src => src.Metadata.Specialty))
                .ForMember(dest => dest.Name, opt => opt.MapFrom(src => src.Hospital.Name))
                .ForMember(dest => dest.Identifiers, opt => opt.MapFrom(src => new List<ModelSchemas.Cda.EntityIdentifier>()
                    {
                        new ModelSchemas.Cda.EntityIdentifier
                        {
                            AssigningAuthorityName = "HPI-O",
                            AssigningGeographicArea = "National Identifier",
                            Root = "1.2.36.1.2001.1003.0." + src.Hospital.HpiO,
                        }
                    }));
        }

        #endregion Methods
    }
}