﻿using System;
using System.Linq;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;
using HIPS.PcehrSchemas;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.PCEHR;
using ChangeHistoryViewRegistry = Nehta.VendorLibrary.PCEHR.GetChangeHistoryView;
using Registry = Nehta.VendorLibrary.PCEHR.DocumentRegistry;

namespace HIPS.CommonBusinessLogic.Pcehr
{
    public static class DocumentHelper
    {
        /// <summary>
        /// *** NOTE:   There are two ExtractMetaData functions that are identical except that they take either Nehta.VendorLibrary.PCEHR.DocumentRegistry
        ///             or the Nehta.VendorLibrary.PCEHR.GetChangeHistoryView.  These are similar classes unfortunately have the same named internal objects
        ///             and no related base class
        /// Extract document meta data from the ExtrinsicObjectType and return a DocumentMetaDataItem
        /// </summary>
        /// <param name="extrinsicObj"></param>
        /// <returns></returns>
        public static DocumentMetaDataItem ExtractMetaData(Registry.ExtrinsicObjectType extrinsicObj, out string IhiNumber)
        {
            DocumentMetaDataItem metaData = new DocumentMetaDataItem();
            char[] caretDelimiter = new char[] { '^' };

            // Get the Document Entry UUID. NOC requires this UUID as the parameter to GetChangeHistoryView rather than the DocumentUniqueId OID.
            metaData.DocumentEntryUuid = extrinsicObj.id;

            //determine the status of the object
            switch (extrinsicObj.status)
            {
                case ("urn:oasis:names:tc:ebxml-regrep:StatusType:Approved"):
                    metaData.DocumentStatusCode = DocumentStatus.Approved;
                    break;

                case ("urn:oasis:names:tc:ebxml-regrep:StatusType:Submitted"):
                    metaData.DocumentStatusCode = DocumentStatus.Submitted;
                    break;

                case ("urn:oasis:names:tc:ebxml-regrep:StatusType:Deprecated"):
                    metaData.DocumentStatusCode = DocumentStatus.Deprecated;
                    break;

                case ("urn:orcl.reg:names:StatusType:Deleted"):
                    metaData.DocumentStatusCode = DocumentStatus.Deleted;
                    break;
            }

            //get author and institutions
            Registry.ClassificationType authorClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.author);
            foreach (Registry.SlotType1 slot in authorClassification.Slot)
            {
                //get the author institution (e.g. DHS5670^^^^^^^^^1.2.36.1.2001.1003.0.8003624900000594)
                if (slot.name.Equals("authorInstitution"))
                {
                    String authorInstitution = slot.ValueList.Value[0];

                    //obtain the HPIO and populate (e.g. DHS5670)
                    metaData.AuthorInstitution = authorInstitution.Substring(authorInstitution.LastIndexOf(".") + 1);

                    //obtain the Author Institution Name and populate (e.g. 8003624900000594)
                    metaData.AuthorInstitutionName = authorInstitution.Substring(0, authorInstitution.IndexOf("^"));
                }

                //get the author person (e.g. ^Smith^John^^^Mr^^^1.2.36.1.2001.1005.41.8003624900000594^&JSMITH01&ISO
                if (slot.name.Equals("authorPerson"))
                {
                    string authorPerson = slot.ValueList.Value[0];

                    string[] person = authorPerson.Split(caretDelimiter, 7);

                    //populate the Given Name (e.g. John)
                    metaData.AuthorPersonGivenName = person[2];

                    //populate the Family Name (e.g. Smith)
                    metaData.AuthorPersonFamilyName = person[1];

                    //populate the Prefix (e.g. Mr)
                    metaData.AuthorPersonNamePrefix = person[5];

                    //populate the fullname (e.g. Mr John Smith)
                    metaData.AuthorPerson = ((metaData.AuthorPersonNamePrefix.Trim() != "") ? metaData.AuthorPersonNamePrefix + " " : "") + metaData.AuthorPersonGivenName + " " + metaData.AuthorPersonFamilyName;
                }
            }

            //get top level slot values
            foreach (Registry.SlotType1 slot in extrinsicObj.Slot)
            {
                //get creation time (e.g. 201207312133)
                if (slot.name.Equals("creationTime"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        metaData.CreationTime = HL7DateTime.Parse(slot.ValueList.Value[0]).Value;

                        //metaData.CreationTime = DateTime.ParseExact(slot.ValueList.Value[0], pcehrDateTimeFormat, CultureInfo.InvariantCulture.DateTimeFormat, DateTimeStyles.None);
                    }
                }

                //get hash (e.g. 143eb6a983f93adbc691df1edd8775624cfd41e3)
                if (slot.name.Equals("hash"))
                {
                    metaData.Hash = slot.ValueList.Value[0];
                }

                //get service Start time (e.g. 201207312133)
                if (slot.name.Equals("serviceStartTime"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        metaData.ServiceStartTime = HL7DateTime.Parse(slot.ValueList.Value[0]).Value;

                        //metaData.ServiceStartTime = DateTime.ParseExact(slot.ValueList.Value[0], pcehrDateTimeFormat, CultureInfo.InvariantCulture.DateTimeFormat, DateTimeStyles.None);
                    }
                }

                //get service Stop time (e.g. 201208081430)
                if (slot.name.Equals("serviceStopTime"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        metaData.ServiceStopTime = HL7DateTime.Parse(slot.ValueList.Value[0]).Value;

                        //metaData.ServiceStopTime = DateTime.ParseExact(slot.ValueList.Value[0], pcehrDateTimeFormat, CultureInfo.InvariantCulture.DateTimeFormat, DateTimeStyles.None);
                    }
                }

                //get size (e.g. 26015)
                if (slot.name.Equals("size"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        int size = 0;
                        if (Int32.TryParse(slot.ValueList.Value[0].ToString(), out size))
                        {
                            metaData.DocumentSize = size;
                        }
                    }
                }

                //get Repository Id e.g. 1.2.36.1.2001.1007.10)
                if (slot.name.Equals("repositoryUniqueId"))
                {
                    metaData.RepositoryUniqueId = slot.ValueList.Value[0];
                }
            }

            //get DocumentClassCode (e.g. "18842-5" "Discharge Summary")
            Registry.ClassificationType classCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.classCode);
            metaData.DocumentClassCode = classCodeClassification.nodeRepresentation;
            metaData.DocumentClassName = classCodeClassification.Name.LocalizedString[0].value;

            //get HealthCareFacilityTypeCode (e.g. "8401" "Hospitals (except Psychiatric Hospitals)")
            Registry.ClassificationType healthCareFacilityTypeCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.healthCareFacilityTypeCode);
            metaData.HealthCareFacilityTypeCode = healthCareFacilityTypeCodeClassification.nodeRepresentation;
            metaData.HealthCareFacilityTypeName = healthCareFacilityTypeCodeClassification.Name.LocalizedString[0].value;

            //get PracticeSettingCode (e.g. "8401-5" "General Hospital")
            Registry.ClassificationType practiceSettingCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.practiceSettingCode);
            metaData.PracticeSettingTypesCode = practiceSettingCodeClassification.nodeRepresentation;
            metaData.PracticeSettingTypesName = practiceSettingCodeClassification.Name.LocalizedString[0].value;

            //get FormatCode (e.g. "1.2.36.1.2001.1006.1.20000.11" "Discharge Summary - Conformance 3A")
            Registry.ClassificationType formatCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.formatCode);
            metaData.FormatCode = formatCodeClassification.nodeRepresentation;
            metaData.FormatName = formatCodeClassification.Name.LocalizedString[0].value;

            //get Unique Document Id (e.g. "1.2.36.1.2001.1005.26.1.1.45363^12343")
            Registry.ExternalIdentifierType uniqueIdExternalIdentifier = extrinsicObj.ExternalIdentifier.FirstOrDefault(element => element.identificationScheme == UuidXDSDocumentEntry.uniqueId);
            metaData.DocumentUniqueId = uniqueIdExternalIdentifier.value;

            //get PatientID - IHI (e.g. "8003608000002519")
            //added to the response object if it is null
            Registry.ExternalIdentifierType patientIdExternalIdentifier = extrinsicObj.ExternalIdentifier.FirstOrDefault(element => element.identificationScheme == UuidXDSDocumentEntry.patientId);
            String patientId = patientIdExternalIdentifier.value;
            if (!patientId.IsNullOrEmptyWhitespace())
            {
                IhiNumber = patientId.Substring(0, patientId.IndexOf("^"));
            }
            else
            {
                IhiNumber = "";
            }

            //add DocumentMetaDataItem to DocumentListReponse list
            return metaData;
        }

        /// <summary>
        /// *** NOTE:   There are two ExtractMetaData functions that are identical except that they take either Nehta.VendorLibrary.PCEHR.DocumentRegistry
        ///             or the Nehta.VendorLibrary.PCEHR.GetChangeHistoryView.  These are similar classes unfortunately have the same named internal objects
        ///             and no related base class
        /// Extract document meta data from the ExtrinsicObjectType and return a DocumentMetaDataItem
        /// </summary>
        /// <param name="extrinsicObj"></param>
        /// <returns></returns>
        public static DocumentMetaDataItem ExtractMetaData(ChangeHistoryViewRegistry.ExtrinsicObjectType extrinsicObj, out string IhiNumber)
        {
            DocumentMetaDataItem metaData = new DocumentMetaDataItem();
            char[] caretDelimiter = new char[] { '^' };

            // Get the Document Entry UUID. NOC requires this UUID as the parameter to GetChangeHistoryView rather than the DocumentUniqueId OID.
            metaData.DocumentEntryUuid = extrinsicObj.id;

            //determine the status of the object
            switch (extrinsicObj.status)
            {
                case ("urn:oasis:names:tc:ebxml-regrep:StatusType:Approved"):
                    metaData.DocumentStatusCode = DocumentStatus.Approved;
                    break;

                case ("urn:oasis:names:tc:ebxml-regrep:StatusType:Submitted"):
                    metaData.DocumentStatusCode = DocumentStatus.Submitted;
                    break;

                case ("urn:oasis:names:tc:ebxml-regrep:StatusType:Deprecated"):
                    metaData.DocumentStatusCode = DocumentStatus.Deprecated;
                    break;

                case ("urn:orcl.reg:names:StatusType:Deleted"):
                    metaData.DocumentStatusCode = DocumentStatus.Deleted;
                    break;
            }

            //get author and institutions
            ChangeHistoryViewRegistry.ClassificationType authorClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.author);
            foreach (ChangeHistoryViewRegistry.SlotType1 slot in authorClassification.Slot)
            {
                //get the author institution (e.g. DHS5670^^^^^^^^^1.2.36.1.2001.1003.0.8003624900000594)
                if (slot.name.Equals("authorInstitution"))
                {
                    String authorInstitution = slot.ValueList.Value[0];

                    //obtain the HPIO and populate (e.g. DHS5670)
                    metaData.AuthorInstitution = authorInstitution.Substring(authorInstitution.LastIndexOf(".") + 1);

                    //obtain the Author Institution Name and populate (e.g. 8003624900000594)
                    metaData.AuthorInstitutionName = authorInstitution.Substring(0, authorInstitution.IndexOf("^"));
                }

                //get the author person (e.g. ^Smith^John^^^Mr^^^1.2.36.1.2001.1005.41.8003624900000594^&JSMITH01&ISO
                if (slot.name.Equals("authorPerson"))
                {
                    string authorPerson = slot.ValueList.Value[0];

                    string[] person = authorPerson.Split(caretDelimiter, 7);

                    //populate the Given Name (e.g. John)
                    metaData.AuthorPersonGivenName = person[2];

                    //populate the Family Name (e.g. Smith)
                    metaData.AuthorPersonFamilyName = person[1];

                    //populate the Prefix (e.g. Mr)
                    metaData.AuthorPersonNamePrefix = person[5];

                    //populate the fullname (e.g. Mr John Smith)
                    metaData.AuthorPerson = ((metaData.AuthorPersonNamePrefix.Trim() != "") ? metaData.AuthorPersonNamePrefix + " " : "") + metaData.AuthorPersonGivenName + " " + metaData.AuthorPersonFamilyName;
                }
            }

            //get top level slot values
            foreach (ChangeHistoryViewRegistry.SlotType1 slot in extrinsicObj.Slot)
            {
                //get creation time (e.g.
                if (slot.name.Equals("creationTime"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        metaData.CreationTime = HL7DateTime.Parse(slot.ValueList.Value[0]).Value;
                    }
                }

                //get hash (e.g. 143eb6a983f93adbc691df1edd8775624cfd41e3)
                if (slot.name.Equals("hash"))
                {
                    metaData.Hash = slot.ValueList.Value[0];
                }

                //get service Start time (e.g. 201207312133)
                if (slot.name.Equals("serviceStartTime"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        metaData.ServiceStartTime = HL7DateTime.Parse(slot.ValueList.Value[0]).Value;
                    }
                }

                //get service Stop time (e.g. 201208081430)
                if (slot.name.Equals("serviceStopTime"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        metaData.ServiceStopTime = HL7DateTime.Parse(slot.ValueList.Value[0]).Value;
                    }
                }

                //get size (e.g. 26015)
                if (slot.name.Equals("size"))
                {
                    if (!slot.ValueList.Value[0].IsNullOrEmptyWhitespace())
                    {
                        int size = 0;
                        if (Int32.TryParse(slot.ValueList.Value[0].ToString(), out size))
                        {
                            metaData.DocumentSize = size;
                        }
                    }
                }

                //get Repository Id e.g. 1.2.36.1.2001.1007.10)
                if (slot.name.Equals("repositoryUniqueId"))
                {
                    metaData.RepositoryUniqueId = slot.ValueList.Value[0];
                }
            }

            //get DocumentClassCode (e.g. "18842-5" "Discharge Summary")
            ChangeHistoryViewRegistry.ClassificationType classCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.classCode);
            metaData.DocumentClassCode = classCodeClassification.nodeRepresentation;
            metaData.DocumentClassName = classCodeClassification.Name.LocalizedString[0].value;

            //get HealthCareFacilityTypeCode (e.g. "8401" "Hospitals (except Psychiatric Hospitals)")
            ChangeHistoryViewRegistry.ClassificationType healthCareFacilityTypeCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.healthCareFacilityTypeCode);
            metaData.HealthCareFacilityTypeCode = healthCareFacilityTypeCodeClassification.nodeRepresentation;
            metaData.HealthCareFacilityTypeName = healthCareFacilityTypeCodeClassification.Name.LocalizedString[0].value;

            //get PracticeSettingCode (e.g. "8401-5" "General Hospital")
            ChangeHistoryViewRegistry.ClassificationType practiceSettingCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.practiceSettingCode);
            metaData.PracticeSettingTypesCode = practiceSettingCodeClassification.nodeRepresentation;
            metaData.PracticeSettingTypesName = practiceSettingCodeClassification.Name.LocalizedString[0].value;

            //get FormatCode (e.g. "1.2.36.1.2001.1006.1.20000.11" "Discharge Summary - Conformance 3A")
            ChangeHistoryViewRegistry.ClassificationType formatCodeClassification = extrinsicObj.Classification.FirstOrDefault(element => element.classificationScheme == UuidXDSDocumentEntry.formatCode);
            metaData.FormatCode = formatCodeClassification.nodeRepresentation;
            metaData.FormatName = formatCodeClassification.Name.LocalizedString[0].value;

            //get Unique Document Id (e.g. "1.2.36.1.2001.1005.26.1.1.45363^12343")
            ChangeHistoryViewRegistry.ExternalIdentifierType uniqueIdExternalIdentifier = extrinsicObj.ExternalIdentifier.FirstOrDefault(element => element.identificationScheme == UuidXDSDocumentEntry.uniqueId);
            metaData.DocumentUniqueId = uniqueIdExternalIdentifier.value;

            //get PatientID - IHI (e.g. "8003608000002519")
            //added to the response object if it is null
            ChangeHistoryViewRegistry.ExternalIdentifierType patientIdExternalIdentifier = extrinsicObj.ExternalIdentifier.FirstOrDefault(element => element.identificationScheme == UuidXDSDocumentEntry.patientId);
            String patientId = patientIdExternalIdentifier.value;
            if (!patientId.IsNullOrEmptyWhitespace())
            {
                IhiNumber = patientId.Substring(0, patientId.IndexOf("^"));
            }
            else
            {
                IhiNumber = "";
            }

            //add DocumentMetaDataItem to DocumentListReponse list
            return metaData;
        }

        /// <summary>
        /// Validates the service certificate.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="certificate">The certificate.</param>
        /// <param name="chain">The chain.</param>
        /// <param name="sslPolicyErrors">The SSL policy errors.</param>
        /// <returns></returns>
        internal static bool ValidateServiceCertificate(object sender, X509Certificate certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            if (sslPolicyErrors == System.Net.Security.SslPolicyErrors.None)
            {
                return true;
            }

            // If there are errors in the certificate chain, look at each error to determine the cause.
            if ((sslPolicyErrors & System.Net.Security.SslPolicyErrors.RemoteCertificateChainErrors) != 0)
            {
                if (chain != null && chain.ChainStatus != null)
                {
                    foreach (System.Security.Cryptography.X509Certificates.X509ChainStatus status in chain.ChainStatus)
                    {
                        if ((certificate.Subject == certificate.Issuer) &&
                           (status.Status == System.Security.Cryptography.X509Certificates.X509ChainStatusFlags.UntrustedRoot))
                        {
                            // Self-signed certificates with an untrusted root are valid.
                            continue;
                        }
                        else
                        {
                            if (status.Status != System.Security.Cryptography.X509Certificates.X509ChainStatusFlags.NoError)
                            {
                                // If there are any other errors in the certificate chain, the certificate is invalid,
                                // so the method returns false.
                                return false;
                            }
                        }
                    }
                }

                // When processing reaches this line, the only errors in the certificate chain are
                // untrusted root errors for self-signed certificates. These certificates are valid
                // for default Exchange Server installations, so return true.
                return true;
            }
            else
            {
                // In all other cases, return false.
                return false;
            }
        }
    }
}