﻿using System.Data.SqlClient;
using HIPS.CommonBusinessLogic;
using HIPS.CommonBusinessLogic.Singleton;
using HIPS.CommonSchemas;
using HIPS.HibIntegration.HL7.DataStructure;
using HIPS.PcehrDataStore.Schemas;

namespace HIPS.HibIntegration.Loader
{
    /// <summary>
    /// Finds and links the patient based on the hospital-based identifier (MRN) and state-wide identifier (SAUHI) provided.
    /// </summary>
    public class PatientLinker
    {
        private Hospital hospital;
        private HospitalPatient hospitalPatient;
        private PatientAccess patientAccess;
        private PatientMaster patientMaster;
        private object syncRoot = new object();
        private SqlTransaction transaction;
        private UserDetails user;

        /// <summary>
        /// Constructs an instance of the patient linking business logic.
        /// </summary>
        /// <param name="user">The user responsible for the action</param>
        /// <param name="transaction">The transaction in which to perform any merge or move that is required.</param>
        public PatientLinker(UserDetails user, SqlTransaction transaction)
        {
            this.patientAccess = new PatientAccess(user);
            this.user = user;
            this.transaction = transaction;
        }

        /// <summary>
        /// Finds the hospital patient and ensures that it is linked to the right patient master.
        /// This may include creating a new patient master, or moving it to a different patient
        /// master, or merging two patient masters. See the diagram "EMPI Linking" for details.
        ///
        /// If the returned patient master is a newly created one, it has not yet been saved to
        /// the database and so the patient master ID is null. Once saved, the patient master ID
        /// needs to be set into the hospital patient object before that object can be saved.
        ///
        /// The hospital is identified using the assigning authority component of the MRN, which
        /// must correspond to the Code value in the HospitalCode table for the CodeSystem of
        /// "pasFacCd".
        /// </summary>
        /// <param name="mrn">The MRN (including the ID and the assigning authority)</param>
        /// <param name="statePatientId">The enterprise ID for the patient (e.g. SAUHI)</param>
        /// <param name="finalHospital">Returns the hospital object</param>
        /// <param name="finalHospitalPatient">Returns the hospital patient (MRN) object</param>
        /// <param name="finalPatientMaster">Returns the patient master (SAUHI) object</param>
        /// <exception cref="HL7MessageErrorException">If the hospital is not found or a database error occurs.</exception>
        public void FindAndLinkPatient(CX mrn, CX statePatientId, out Hospital finalHospital, out HospitalPatient finalHospitalPatient, out PatientMaster finalPatientMaster)
        {
            lock (syncRoot)
            {
                FindHospital(mrn);
                if (FindPatientByMrn(mrn))
                {
                    GetPatientMaster();
                    if (statePatientId == null || string.IsNullOrEmpty(statePatientId.ID))
                    {
                        // The incoming message has no SAUHI; no linking to do.
                    }
                    else if (patientMaster.StatePatientId == statePatientId.ID)
                    {
                        // The incoming message has the same SAUHI as the existing patient; no linking to do.
                    }
                    else if (string.IsNullOrEmpty(patientMaster.StatePatientId))
                    {
                        MergeIntoEnterpriseRecord(statePatientId);
                    }
                    else
                    {
                        MoveToEnterpriseRecord(statePatientId);
                    }
                }
                else
                {
                    NewHospitalPatient(mrn, statePatientId);
                }
                finalHospital = hospital;
                finalHospitalPatient = hospitalPatient;
                finalPatientMaster = patientMaster;
            }
        }

        /// <summary>
        /// Finds the hospital from the assigning authority of the MRN.
        /// </summary>
        /// <param name="mrn">The MRN</param>
        /// <exception cref="HL7MessageErrorException">If the hospital is not found.</exception>
        private void FindHospital(CX mrn)
        {
            if (mrn == null || mrn.assigningauthority == null)
            {
                throw new HL7MessageErrorException(ConstantsResource.NoMrnForPatient);
            }
            string codeSystemName = ConstantsResource.PatientAdministrationSystemHospitalCodeSystem;
            this.hospital = HospitalSingleton.Value.Find(mrn.assigningauthority.namespaceID, codeSystemName);
            if (this.hospital == null)
            {
                string description = string.Format(ConstantsResource.CannotFindHospital, mrn.assigningauthority.namespaceID, codeSystemName);
                throw new HL7MessageErrorException(description);
            }
        }

        /// <summary>
        /// Finds the patient by MRN.
        /// </summary>
        /// <returns>Whether found.</returns>
        /// <exception cref="HL7MessageErrorException">If the database access fails</exception>
        private bool FindPatientByMrn(CX mrn)
        {
            HipsResponse response = patientAccess.HospitalPatientDataAccess.Get(hospital.HospitalId.Value, mrn.ID, out hospitalPatient, transaction);
            if (response.Status == HipsResponseIndicator.OK)
            {
                return true;
            }
            else if (response.Status == HipsResponseIndicator.DatabaseError)
            {
                throw new HL7MessageErrorException(response.HipsErrorMessage);
            }
            return false;
        }

        /// <summary>
        /// Finds the patient by state patient ID.
        /// </summary>
        /// <returns>Whether found</returns>
        /// <exception cref="HL7MessageErrorException">If the database access fails</exception>
        private bool FindPatientByStatePatientId(CX statePatientId)
        {
            HipsResponse response = patientAccess.PatientMasterDataAccess.GetByStatePatientId(statePatientId.ID, hospital.HealthProviderOrganisationNetworkId, out patientMaster, transaction);
            if (response.Status == HipsResponseIndicator.OK)
            {
                return true;
            }
            else if (response.Status == HipsResponseIndicator.DatabaseError)
            {
                throw new HL7MessageErrorException(response.HipsErrorMessage);
            }
            return false;
        }

        /// <summary>
        /// Gets the patient master for the hospital patient found.
        /// </summary>
        /// <returns>Whether found.</returns>
        /// <exception cref="HL7MessageErrorException">If the database access fails</exception>
        private bool GetPatientMaster()
        {
            HipsResponse response = patientAccess.PatientMasterDataAccess.Get(hospitalPatient.PatientMasterId, hospital.HealthProviderOrganisationNetworkId, out patientMaster, transaction);
            if (response.Status == HipsResponseIndicator.OK)
            {
                return true;
            }
            else if (response.Status == HipsResponseIndicator.DatabaseError)
            {
                throw new HL7MessageErrorException(response.HipsErrorMessage);
            }
            return false;
        }

        /// <summary>
        /// Handles the situation when an MRN that did not previously have a
        /// SAUHI is assigned a SAUHI. This should only happen when the scope
        /// of EMPI is being expanded to cover additional hospitals.
        ///
        /// If there is a patient master with the given SAUHI, then we merge
        /// the existing MRN into that patient master. If there is no patient
        /// master with the given SAUHI, then we assign the SAUHI to the
        /// existing patient master.
        /// </summary>
        private void MergeIntoEnterpriseRecord(CX statePatientId)
        {
            PatientMaster oldPatientMaster = patientMaster;

            // SAUHI has just been allocated to this record.
            if (FindPatientByStatePatientId(statePatientId))
            {
                // Merge old patient master into found one.
                hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                if (!patientAccess.PatientMasterDataAccess.Merge(oldPatientMaster, patientMaster, hospital.HealthProviderOrganisationNetworkId, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.PatientMasterDataAccess.GetType().FullName));
                }
            }
            else
            {
                // The SAUHI was not found; minimise disruption by just assigning it to the existing patient master.
                patientMaster = oldPatientMaster;
                patientMaster.StatePatientId = statePatientId.ID;
            }
        }

        /// <summary>
        /// Handles the situation when an MRN suddenly moves from one SAUHI to another.
        /// This should only happen when a message from PAS gets ahead of the A43 from
        /// EMPI that should make this change.
        ///
        /// Notes:
        /// Also moves any other MRNs from the same hospital (previously merged).
        /// Also removes the IHI from the old patient master if no MRNs left on it.
        /// </summary>
        private void MoveToEnterpriseRecord(CX statePatientId)
        {
            PatientMaster oldPatientMaster = patientMaster;

            if (FindPatientByStatePatientId(statePatientId))
            {
                // Move this MRN to the other patient master.
                HipsResponse response = patientAccess.HospitalPatientDataAccess.Move(hospitalPatient, patientMaster, transaction);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    throw new HL7MessageErrorException(response.HipsErrorMessage);
                }
            }
            else
            {
                // The patient is being unlinked to a new SAUHI. Create a temporary
                // patient master by copying all of the demographic information
                // then do the move.
                patientMaster = oldPatientMaster.CloneToNewStatePatientId(statePatientId.ID);

                if (!patientAccess.PatientMasterDataAccess.Insert(patientMaster, transaction))
                {
                    throw new HL7MessageErrorException(string.Format(ConstantsResource.DatabaseError, patientAccess.PatientMasterDataAccess.GetType().FullName));
                }
                HipsResponse response = patientAccess.HospitalPatientDataAccess.Move(hospitalPatient, patientMaster, transaction);
                if (response.Status != HipsResponseIndicator.OK)
                {
                    throw new HL7MessageErrorException(response.HipsErrorMessage);
                }
            }
        }

        /// <summary>
        /// Create a new hospital patient and link it either to an existing patient master or a new patient master.
        /// </summary>
        private void NewHospitalPatient(CX mrn, CX statePatientId)
        {
            hospitalPatient = new HospitalPatient();
            hospitalPatient.Mrn = mrn.ID;
            hospitalPatient.HospitalId = hospital.HospitalId.Value;

            if (statePatientId != null && !string.IsNullOrEmpty(statePatientId.ID))
            {
                if (FindPatientByStatePatientId(statePatientId))
                {
                    // Link the new MRN to the existing patient master
                    hospitalPatient.PatientMasterId = patientMaster.PatientMasterId.Value;
                }
                else
                {
                    // Create a new patient master and store the SAUHI in it.
                    patientMaster = new PatientMaster();
                    patientMaster.StatePatientId = statePatientId.ID;
                }
            }
            else
            {
                // There was no SAUHI in the message, just create a new patient master.
                patientMaster = new PatientMaster();
            }
        }
    }
}