﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using HIPS.Common.DataStore.DataAccess;
using HIPS.Common.PcehrDataStore.DataAccess;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;

namespace HIPS.PcehrDataStore.DataAccess
{
    /// <summary>
    /// This class allows access to the PcehrMessageQueue table
    /// </summary>
    public class PcehrMessageQueueDl : BaseDl
    {
        #region Constructors

        public PcehrMessageQueueDl(UserDetails user)
            : base(user)
        {
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Gets a count of all pending operations for the specified sourceSystemSetId ID that are before the specific date,
        /// excluding the current PcehrMessageQueue Item
        /// </summary>
        /// <param name="sourceSystemSetId">SourceSystemSetId for the queue item</param>
        /// <param name="CreatedDateTime">Created DateTime of the current queue item</param>
        /// <param name="pcehrMessageQueueId">The PCEHR message queue ID</param>
        /// <returns>Count of items, 0 if none</returns>
        public int CountEarlierPendingMessages(string sourceSystemSetId, DateTime createdDateTime, int? pcehrMessageQueueId)
        {
            int count = 0;
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueCountPending"))
                {
                    command.Parameters.Add(new SqlParameter("@SourceSystemSetId", sourceSystemSetId));
                    command.Parameters.Add(new SqlParameter("@CreatedDateTime", createdDateTime));
                    command.Parameters.Add(new SqlParameter("@PcehrMessageQueueId", pcehrMessageQueueId));

                    SqlDataReader reader = command.ExecuteReader();
                    while (reader.Read())
                    {
                        count = reader.GetInt32(0);
                    }
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format(ConstantsResource.ErrorMessagePcehrMessageQueueCountPending, sourceSystemSetId, createdDateTime);
                EventLogger.WriteLog(msg, ex, User, LogMessage.HIPS_MESSAGE_135);
                count = 0;
            }
            return count;
        }

        /// <summary>
        /// Deletes the specified PCEHR message queue record.
        /// </summary>
        /// <param name="pcehrMessageQueue">The PCEHR message queue record.</param>
        /// <param name="connection">The existing database connection to use (optional)</param>
        /// <param name="transaction">The existing database transaction to use (optional)</param>
        /// <returns>Whether the record was inserted</returns>
        public bool Delete(PcehrMessageQueue pcehrMessageQueue, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueDelete", transaction))
            {
                return base.Delete<PcehrMessageQueue>(pcehrMessageQueue, command);
            }
        }

        /// <summary>
        /// Gets a queued operation for the specified PCEHR message queue ID.
        /// </summary>
        /// <param name="pcehrMessageQueueId">The PCEHR message queue ID</param>
        /// <param name="item">The queued operation</param>
        /// <returns>True if an item was found</returns>
        public bool Get(int pcehrMessageQueueId, out PcehrMessageQueue item)
        {
            bool result;
            item = new PcehrMessageQueue();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueGet"))
                {
                    command.Parameters.Add(new SqlParameter("@PcehrMessageQueueId", pcehrMessageQueueId));
                    result = PopulateBusinessObject<PcehrMessageQueue>(command.ExecuteReader(), item);
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format(ConstantsResource.ErrorMessagePcehrMessageQueueGet, pcehrMessageQueueId);
                EventLogger.WriteLog(msg, ex, User, LogMessage.HIPS_MESSAGE_053);
                result = false;
            }
            return result;
        }

        /// <summary>
        /// Gets all the queued operations for the specified episode.
        /// </summary>
        /// <param name="episodeId">The episode ID</param>
        /// <returns>A list of queued operations for the specified episode</returns>
        public List<PcehrMessageQueue> GetAll(int episodeId)
        {
            List<PcehrMessageQueue> results = new List<PcehrMessageQueue>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueGet"))
                {
                    command.Parameters.Add(new SqlParameter("@EpisodeId", episodeId));
                    results = GetPopulatedBusinessList<PcehrMessageQueue>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format(ConstantsResource.ErrorMessagePcehrMessageQueueGetAll, episodeId);
                EventLogger.WriteLog(msg, ex, User, LogMessage.HIPS_MESSAGE_052);
            }
            return results;
        }

        /// <summary>
        /// Gets the list of queued operation for the specified dates.
        /// </summary>
        /// <param name="DateTimeFrom">The date time from.</param>
        /// <param name="DateTimeTo">The date time to.</param>
        /// <param name="itemList">The MessageQueueItem list.</param>
        /// <returns>
        /// True if an item was found without error
        /// </returns>
        public IList<MessageQueueItem> GetQueuedOperations(DateTime dateTimeFrom, DateTime dateTimeTo)
        {
            IList<MessageQueueItem> messageQueueItemList = new List<MessageQueueItem>();
            try
            {
                using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueOperationList"))
                {
                    command.Parameters.Add(new SqlParameter("@DateTimeFrom", dateTimeFrom));
                    command.Parameters.Add(new SqlParameter("@DateTimeTo", dateTimeTo));
                    messageQueueItemList = GetPopulatedBusinessList<MessageQueueItem>(command.ExecuteReader());
                    command.Connection.Close();
                }
            }
            catch (Exception ex)
            {
                string msg = string.Format(ConstantsResource.ErrorMessagePcehrMessageQueueDatesGet, dateTimeFrom, dateTimeTo);
                EventLogger.WriteLog(msg, ex, User, LogMessage.HIPS_MESSAGE_053);
            }
            return messageQueueItemList;
        }

        /// <summary>
        /// Inserts the specified PCEHR message queue record.
        /// </summary>
        /// <param name="pcehrMessageQueue">The PCEHR message queue record.</param>
        /// <param name="connection">The existing database connection to use (optional)</param>
        /// <param name="transaction">The existing database transaction to use (optional)</param>
        /// <returns>Whether the record was inserted</returns>
        public bool Insert(PcehrMessageQueue pcehrMessageQueue, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueInsert", transaction))
            {
                return base.Insert<PcehrMessageQueue>(pcehrMessageQueue, command);
            }
        }

        /// <summary>
        /// Updates the specified PCEHR message queue record.
        /// </summary>
        /// <param name="pcehrMessageQueue">The PCEHR message queue record.</param>
        /// <param name="connection">The existing database connection to use (optional)</param>
        /// <param name="transaction">The existing database transaction to use (optional)</param>
        /// <returns>Whether the record was inserted</returns>
        public bool Update(PcehrMessageQueue pcehrMessageQueue, SqlTransaction transaction = null)
        {
            using (SqlCommand command = GetSqlCommand("hips.PcehrMessageQueueUpdate", transaction))
            {
                return base.Update<PcehrMessageQueue>(pcehrMessageQueue, command);
            }
        }

        #endregion Methods
    }
}