﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using HIPS.Base.Schemas;
using HIPS.Base.Schemas.Enumerators;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrDataStore.Schemas.Schemas;

using BaseEnumerators = HIPS.Base.Schemas.Enumerators;

namespace HIPS.PcehrDataStore.Schemas
{
    /// <summary>
    /// This class represents a Patient Master.
    /// </summary>
    [KnownType(typeof(PatientMaster))]
    [Serializable]
    [DataContract]
    public class PatientMaster : BaseSchema
    {
        #region Fields

        private int currentSexId;
        private DateTime dateOfBirth;
        private DateTime? dateOfDeath;
        private string deathIndicatorDesc;
        private int? deathIndicatorId;
        private string dvaNumber;
        private string ihi;
        private DateTime? ihiLastValidated;
        private string ihiRecordStatus;
        private int ihiRecordStatusId;
        private string ihiStatus;
        private int ihiStatusId;
        private bool? isMedicareNumberValid;
        private string medicareIrn;
        private string medicareNumber;
        private List<PatientMasterName> names;
        private DateTime registeredDateOfBirth;
        private string registeredFamilyName;
        private string registeredGivenName;
        private int registeredSexId;
        private int healthProviderOrganisationNetworkId;

        #endregion Fields

        #region Construtors

        /// <summary>
        /// Initialises a new instance of the <see cref="PatientMaster" /> class.
        /// </summary>
        public PatientMaster()
        {
            Names = new List<PatientMasterName>();
            Addresses = new List<Address>();
            Contacts = new List<Contact>();
        }

        #endregion Construtors

        #region Properties

        /// <summary>
        /// Gets or sets the addresses.
        /// </summary>
        /// <value>
        /// The addresses.
        /// </value>
        [DataMember]
        public List<Address> Addresses { get; set; }

        /// <summary>
        /// Gets or sets the contacts.
        /// </summary>
        /// <value>
        /// The contacts.
        /// </value>
        [DataMember]
        public List<Contact> Contacts { get; set; }

        /// <summary>
        /// Gets the patient's current name.
        /// </summary>
        /// <value>
        /// The name of the current.
        /// </value>
        public PatientMasterName CurrentName
        {
            get
            {
                PatientMasterName currentName = GetName(NameTypes.Current);
                if (currentName == null)
                {
                    currentName = new PatientMasterName();
                    currentName.NameTypeId = (int)NameTypes.Current;
                }
                return currentName;
            }
        }

        /// <summary>
        /// Gets or sets the current sex id.
        /// </summary>
        /// <value>
        /// The current sex id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int CurrentSexId
        {
            get
            {
                return this.currentSexId;
            }
            set
            {
                Set(() => CurrentSexId, ref this.currentSexId, value);
            }
        }

        /// <summary>
        /// Gets or sets the date of birth.
        /// </summary>
        /// <value>
        /// The date of birth.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public DateTime DateOfBirth
        {
            get
            {
                return this.dateOfBirth;
            }
            set
            {
                Set(() => DateOfBirth, ref this.dateOfBirth, value);
            }
        }

        /// <summary>
        /// Gets or sets the date of death.
        /// </summary>
        /// <value>
        /// The date of death.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public DateTime? DateOfDeath
        {
            get
            {
                return this.dateOfDeath;
            }
            set
            {
                Set(() => DateOfDeath, ref this.dateOfDeath, value);
            }
        }

        /// <summary>
        /// Gets or sets the current Death Indicator Desc.
        /// As it is not stored and just linked to an enum it does not have a DataBaseInfoAttributes
        /// </summary>
        /// <value>
        /// The current death indicator description
        /// </value>
        [DataMember]
        public string DeathIndicatorDesc
        {
            get
            {
                return this.deathIndicatorDesc;
            }
            set
            {
                Set(() => DeathIndicatorDesc, ref this.deathIndicatorDesc, value);
            }
        }

        /// <summary>
        /// Gets or sets the current Death Indicator Id.
        /// </summary>
        /// <value>
        /// The current death indicator
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.Data)]
        public int? DeathIndicatorId
        {
            get
            {
                return this.deathIndicatorId;
            }
            set
            {
                Set(() => DeathIndicatorId, ref this.deathIndicatorId, value);
                if (DeathIndicatorId != null)
                {
                    Set(() => DeathIndicatorDesc, ref this.deathIndicatorDesc, System.Enum.GetName(typeof(DeathIndicator), this.deathIndicatorId));
                }
                else
                {
                    Set(() => DeathIndicatorDesc, ref this.deathIndicatorDesc, null);
                }
            }
        }

        /// <summary>
        /// Gets or sets the DVA file number.
        /// </summary>
        /// <value>
        /// The DVA file number.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string DvaNumber
        {
            get
            {
                return this.dvaNumber;
            }
            set
            {
                Set(() => DvaNumber, ref this.dvaNumber, value);
            }
        }

        /// <summary>
        /// Gets or sets the ihi.
        /// </summary>
        /// <value>
        /// The ihi.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string Ihi
        {
            get
            {
                return this.ihi;
            }
            set
            {
                Set(() => Ihi, ref this.ihi, value);
            }
        }

        /// <summary>
        /// Gets or sets the ihi last validated.
        /// </summary>
        /// <value>
        /// The ihi last validated.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public DateTime? IhiLastValidated
        {
            get
            {
                return this.ihiLastValidated;
            }
            set
            {
                Set(() => IhiLastValidated, ref this.ihiLastValidated, value);
            }
        }

        /// <summary>
        /// Gets or sets the IHI record status description.
        /// </summary>
        /// <value>
        /// The ihi record status description.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string IhiRecordStatus
        {
            get
            {
                return this.ihiRecordStatus;
            }
            set
            {
                Set(() => IhiRecordStatus, ref this.ihiRecordStatus, value);
            }
        }

        /// <summary>
        /// Gets or sets the IHI record status ID.
        /// </summary>
        /// <value>
        /// The IHI record status ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public int IhiRecordStatusId
        {
            get
            {
                return this.ihiRecordStatusId;
            }
            set
            {
                Set(() => IhiRecordStatusId, ref this.ihiRecordStatusId, value);
            }
        }

        /// <summary>
        /// Gets or sets the IHI status description.
        /// </summary>
        /// <value>
        /// The IHI status description.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public string IhiStatus
        {
            get
            {
                return this.ihiStatus;
            }
            set
            {
                Set(() => IhiStatus, ref this.ihiStatus, value);
            }
        }

        /// <summary>
        /// Gets or sets the ihi status ID.
        /// </summary>
        /// <value>
        /// The ihi status ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public int IhiStatusId
        {
            get
            {
                return this.ihiStatusId;
            }
            set
            {
                Set(() => IhiStatusId, ref this.ihiStatusId, value);
            }
        }

        /// <summary>
        /// Gets or sets whether the Medicare card number is valid.
        /// </summary>
        /// <value>
        /// Whether the Medicare card number is valid.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public bool? IsMedicareNumberValid
        {
            get
            {
                return this.isMedicareNumberValid;
            }
            set
            {
                Set(() => IsMedicareNumberValid, ref this.isMedicareNumberValid, value);
            }
        }

        /// <summary>
        /// Gets the name that is registered with Medicare HI Service and associated with the currently assigned IHI.
        /// </summary>
        /// <value>
        /// The registered name.
        /// </value>
        public PatientMasterName LegalName
        {
            get
            {
                return new PatientMasterName(null, null, RegisteredGivenName, RegisteredFamilyName, null, null, -1, null);
            }
        }

        /// <summary>
        /// Gets or sets the individual reference number for this patient on the Medicare card (1 digit).
        /// </summary>
        /// <value>
        /// The Medicare individual reference number.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string MedicareIrn
        {
            get
            {
                return this.medicareIrn;
            }
            set
            {
                Set(() => MedicareIrn, ref this.medicareIrn, value);
            }
        }

        /// <summary>
        /// Gets or sets the Medicare card number (10 digits including check digit).
        /// </summary>
        /// <value>
        /// The Medicare card number.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string MedicareNumber
        {
            get
            {
                return this.medicareNumber;
            }
            set
            {
                Set(() => MedicareNumber, ref this.medicareNumber, value);
            }
        }

        /// <summary>
        /// Gets or sets the names.
        /// </summary>
        /// <value>
        /// The names.
        /// </value>
        [DataMember]
        public List<PatientMasterName> Names
        {
            get
            {
                return this.names;
            }
            set
            {
                Set(() => Names, ref this.names, value);
            }
        }

        /// <summary>
        /// Gets or sets the patient master ID.
        /// </summary>
        /// <value>
        /// The patient master ID.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(DatabaseColumnType.AutoGeneratedKey)]
        public int? PatientMasterId
        {
            get
            {
                return base.Id;
            }
            set
            {
                base.Id = value;
            }
        }

        /// <summary>
        /// Gets the previous names.
        /// </summary>
        public List<PatientMasterName> PreviousNames
        {
            get
            {
                return Names.FindAll(result => result.NameTypeId == (int)NameTypes.Previous);
            }
        }

        /// <summary>
        /// Gets or sets the ihi last validated.
        /// </summary>
        /// <value>
        /// The ihi last validated.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public DateTime RegisteredDateOfBirth
        {
            get
            {
                return this.registeredDateOfBirth;
            }
            set
            {
                Set(() => RegisteredDateOfBirth, ref this.registeredDateOfBirth, value);
            }
        }

        /// <summary>
        /// Gets or sets the registered family name.
        /// </summary>
        /// <value>
        /// The name of the registered family.
        /// </value>
        [DataMember]
        public string RegisteredFamilyName
        {
            get
            {
                return this.registeredFamilyName;
            }
            set
            {
                Set(() => RegisteredFamilyName, ref this.registeredFamilyName, value);
            }
        }

        /// <summary>
        /// Gets or sets the registered given name.
        /// </summary>
        /// <value>
        /// The name of the registered given.
        /// </value>
        [DataMember]
        public string RegisteredGivenName
        {
            get
            {
                return this.registeredGivenName;
            }
            set
            {
                Set(() => RegisteredGivenName, ref this.registeredGivenName, value);
            }
        }

        /// <summary>
        /// Gets or sets the registered sex id.
        /// </summary>
        /// <value>
        /// The registered sex id.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.ReadOnlyData)]
        public int RegisteredSexId
        {
            get
            {
                return this.registeredSexId;
            }
            set
            {
                Set(() => RegisteredSexId, ref this.registeredSexId, value);
            }
        }

        /// <summary>
        /// Gets or sets the state/territory's unique patient identifier, such as the SAUHI or HCID.
        /// </summary>
        /// <value>
        /// The the state/territory patient identifier.
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public string StatePatientId { get; set; }

        /// <summary>
        /// Specifies which HPO network this organisation belongs to.
        /// </summary>
        /// <value>
        /// Health Provider Organisation Network Id
        /// </value>
        [DataMember]
        [DataBaseInfoAttributes(BaseEnumerators.DatabaseColumnType.Data)]
        public int HealthProviderOrganisationNetworkId
        {
            get
            {
                return this.healthProviderOrganisationNetworkId;
            }
            set
            {
                Set(() => HealthProviderOrganisationNetworkId, ref this.healthProviderOrganisationNetworkId, value);
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Creates a new patient master that is a clone of this patient master, with a new state/territory patient identifier.
        /// </summary>
        /// <param name="newStatePatientId">The new state/territory patient identifier.</param>
        /// <returns>The new patient master.</returns>
        public PatientMaster CloneToNewStatePatientId(string newStatePatientId)
        {
            PatientMaster clone = this.MemberwiseClone() as PatientMaster;
            clone.PatientMasterId = null;
            clone.StatePatientId = newStatePatientId;
            clone.Addresses = this.Addresses.Select(a => a.CloneForNewPatientMaster()).ToList();
            clone.Contacts = this.Contacts.Select(a => a.CloneForNewPatientMaster()).ToList();
            clone.Names = this.Names.Select(a => a.CloneForNewPatientMaster()).ToList();
            return clone;
        }

        /// <summary>
        /// Checks whether the date of birth and the registered date of birth are equal.
        /// </summary>
        /// <returns>True if they are equal, otherwise false.</returns>
        public bool IsDOBandRDOBEqual()
        {
            bool isDobEqual = true;

            if (this.DateOfBirth != this.RegisteredDateOfBirth)
            {
                isDobEqual = false;
            }

            return isDobEqual;
        }

        /// <summary>
        /// Sets the new name.
        /// </summary>
        /// <param name="titleId">The title id.</param>
        /// <param name="firstNames">The first names.</param>
        /// <param name="lastName">The last name.</param>
        /// <param name="suffixId">The suffix id.</param>
        /// <returns>Whether the name was updated.</returns>
        public bool SetNewCurrentName(int? titleId, string firstNames, string lastName, int? suffixId)
        {
            bool updated = false;
            int nameTypeId = (int)NameTypes.Current;
            PatientMasterName name = new PatientMasterName(titleId, null, firstNames, lastName, suffixId, null, nameTypeId, null);
            PatientMasterName existingName = this.Names.Find(result => (result.CompareKey == name.CompareKey));
            PatientMasterName oldCurrent = Names.Find(result => result.NameTypeId == nameTypeId);
            if (existingName != null)
            {
                //if the existing name was not a NameType of Current then it has been updated
                if (existingName.NameTypeId != nameTypeId)
                {
                    updated = true;
                }
                existingName.NameTypeId = nameTypeId;
            }
            else
            {
                Names.Add(name);
                //if the name has been added then this is an update
                updated = true;
            }

            if ((oldCurrent != null) && (oldCurrent.CompareKey != name.CompareKey))
            {
                oldCurrent.NameTypeId = (int)NameTypes.Previous;
                oldCurrent.NameType = string.Empty;
                //if we are setting the old current name to previous it is an update
                updated = true;
            }
            return updated;
        }

        /// <summary>
        /// Gets the name.
        /// </summary>
        /// <param name="type">The type.</param>
        /// <returns>The name.</returns>
        private PatientMasterName GetName(NameTypes type)
        {
            return Names.FirstOrDefault(result => result.NameTypeId == (int)type);
        }

        #endregion Methods
    }
}