﻿using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Xml;
using HIPS.PcehrDataStore.Schemas;
using Ionic.Zip;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.CdaPackaging.BaseCdaPackage
{
    /// <summary>
    /// Conformance Test Specification: CDA Packaging
    ///                                 Version 1.4 — 12 September 2012
    /// Profile Name:	Base CDA Package
    /// Test Case IDs:   PKG_CDA_002 and PKG_CDA_004
    /// Objective:
    /// To check that the base CDA package shall contain one and only one root entry.
    /// </summary>
    [TestClass]
    public class BaseCdaPackage : CcaTest
    {
        private CdaDocument cdaDocument;
        private ClinicalDocumentVersion version;

        /// <summary>
        /// Shared helper method to ensure that a document has been uploaded
        /// and get the CDA package and document.
        /// The document will be reused between the tests in this class.
        /// </summary>
        private void UploadDocument()
        {
            SharedPackage shared = SharedPackage.GetSharedInstance(this, SampleDocumentType.DischargeSummary3A);
            this.cdaDocument = shared.CdaDocument;
            this.version = shared.Version;
        }

        /// <summary>
        /// Verify that the base CDA package contains one and only one root entry.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_002()
        {
            UploadDocument();
            ICollection<ZipEntry> entries = ZipFile.Read(new MemoryStream(version.Package)).Entries;
            LogAssert.AreEqual(1,
                entries.Count(a => a.FileName.EndsWith("CDA_ROOT.XML")),
                DialogueResource.NumberOfRootEntriesInPackage);
        }

        /// <summary>
        /// Verify that each atomic packaged attachment in the base CDA package
        /// corresponds to a distinct part in the base CDA package.  This is
        /// done by examining the CDA Document and verifying that the file name
        /// has been referenced in a reference element.
        ///
        /// NEHTA feedback is that this must be a case-sensitive check.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA")]
        public void CdaPackaging_PKG_CDA_004()
        {
            UploadDocument();

            // Get a list of all the URIs that are referenced in observationMedia elements in the CDA document.
            XmlNodeList allReferences = cdaDocument.GetAllAttachmentReferences();
            List<string> referenceUris = new List<string>();
            foreach (XmlNode reference in cdaDocument.GetAllAttachmentReferences())
            {
                referenceUris.Add(reference.Attributes["value"].Value);
            }

            // Check that every attachment file in the package is referenced by an observationMedia element in the CDA document.
            List<ZipEntry> attachments = CdaPackage.GetAttachmentEntries(version.Package);
            foreach (ZipEntry attachment in attachments)
            {
                string attachmentFileName = attachment.FileName.Split('/').Last();
                LogAssert.IsTrue(referenceUris.Contains(attachmentFileName),
                    string.Format(DialogueResource.PackagedAttachmentIsReferencedByCdaDocument, attachmentFileName),
                    string.Format(DialogueResource.PackagedAttachmentIsNotReferencedByCdaDocument, attachmentFileName));
            }
        }
    }
}