﻿using System;
using System.IO;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;
using System.Security.Cryptography.Xml;
using System.Xml;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.CdaPackaging.eSignature
{
    /// <summary>
    /// Conformance Test Specification: CDA Packaging
    ///                                 Version 1.4 — 12 September 2012
    /// Profile Name:	eSignature
    /// Group:          XSP signed Container
    /// Objective:
    /// To check that the “eSignature” is an XML document that conforms to a
    /// Signed Container as defined by [ATS 5821—2010] with the root element
    /// of that XML document being the sp:signedPayload element.
    /// </summary>
    [TestClass]
    public class XspSignedContainer_1A : CcaTest
    {
        private CdaDocument cdaDocument;
        private CdaSignature cdaSignature;
        private ClinicalDocumentVersion version;

        /// <summary>
        /// Shared helper method to ensure that a document has been uploaded
        /// and get the CDA signature, CDA document and the package.
        /// The document will be reused between the tests in this class.
        /// </summary>
        private void UploadDocument()
        {
            SharedPackage shared = SharedPackage.GetSharedInstance1A(this, SampleDocumentType.DischargeSummary1A);
            this.cdaDocument = shared.CdaDocument;
            this.cdaSignature = shared.CdaSignature;
            this.version = shared.Version;
        }

        /// <summary>
        /// Verify the XML Document is valid against the Signed Payload XML Schema.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SCP_000()
        {
            UploadDocument();
            bool isValid = cdaSignature.ValidateCdaSignature(true, false);
            LogAssert.IsTrue(isValid, DialogueResource.CdaSignIsValidSignedPayload, DialogueResource.CdaSignIsNotValidSignedPayload);
        }

        /// <summary>
        /// Verify that the id attribute is a unique value within the XML document.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SCP_001()
        {
            UploadDocument();
            string idValue = cdaSignature.GetSignedPayloadDataId();
            int? idCount = cdaSignature.CountSignedPayloadDataId();
            bool isValid = cdaSignature.ValidateCdaSignature(true, false);
            LogAssert.IsTrue(isValid, DialogueResource.CdaSignIsValidSignedPayload, DialogueResource.CdaSignIsNotValidSignedPayload);

            bool idUnique = idCount != null ? (idCount == 1 ? true : false) : false;
            LogAssert.IsTrue(idUnique, DialogueResource.IdAttributeIsUnqiueInXml, DialogueResource.IdAttributeIsNotUnqiueInXml);

            Log(DialogueResource.ValidationProcessIncludesCheckingUniquenessConstraints);
        }

        /// <summary>
        /// Verify that the sp:signedPayloadData element is the only element signed by all of the signatures
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SCP_002()
        {
            UploadDocument();

            // Check that there is only one item referenced by signatures
            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInSignedInfo();
            LogAssert.AreEqual(1, referenceElements.Count, DialogueResource.NumberOfCdaReferenceElements);

            // Check that the item referenced by the signature is the sp:signedPayloadData element.
            string referenceUri = cdaSignature.GetReferenceUri(referenceElements[0]);
            string signedPayloadDataId = cdaSignature.GetSignedPayloadDataId();
            string expectedUri = string.Format("#{0}", signedPayloadDataId);
            LogAssert.AreEqual(expectedUri, referenceUri, DialogueResource.UriAttributeOfReferenceElement);
        }

        /// <summary>
        /// Verify that the eSignature is valid against the XML Signature Schema.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_000()
        {
            UploadDocument();
            bool isValid = cdaSignature.ValidateCdaSignature(false, true);
            LogAssert.IsTrue(isValid,
                passMessage: DialogueResource.ESignatureIsValidSignature,
                failMessage: DialogueResource.ESignatureIsNotValidSignature);
        }

        /// <summary>
        /// Verify that a detached signature is used by checking that the
        /// signature element references, either by URI or transform, a
        /// separate XML document or a sibling element within the same XML
        /// document.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_001()
        {
            UploadDocument();
            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInSignedInfo();
            string referenceUri = cdaSignature.GetReferenceUri(referenceElements[0]);
            LogAssert.IsTrue(referenceUri.StartsWith("#"),
                DialogueResource.SignatureElementReferencesSiblingElementByUri,
                DialogueResource.SignatureElementDoesNotReferenceSiblingElementByUri);
        }

        /// <summary>
        /// Verify that the Exclusive XML Canonicalization method was used on
        /// the signed contents in the ds:SignedInfo element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_002()
        {
            UploadDocument();
            byte[] exc = cdaSignature.GetExclusiveCanonicalisationOfSignedPayloadData();
            string hashOfExclusiveC14n = cdaSignature.CalculateBase64OfSha1Hash(exc);
            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInSignedInfo();
            string digestValue = cdaSignature.GetReferenceDigestValue(referenceElements[0]);
            LogAssert.AreEqual(hashOfExclusiveC14n, digestValue,
                DialogueResource.DigestValueOfSignatureReference,
                DialogueResource.HashOfExclusiveXmlCanonicalisation);
        }

        /// <summary>
        /// Verify that the Algorithm attribute set of the ds:CanonicalizationMethod
        /// element is set to "http://www.w3.org/2001/10/xml-exc-c14n#"
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_003()
        {
            UploadDocument();
            string algorithm = cdaSignature.GetCanonicalizationMethodAlgorithm();
            LogAssert.AreEqual("http://www.w3.org/2001/10/xml-exc-c14n#", algorithm,
                DialogueResource.CanonicalizationMethodAlgorithm);
        }

        /// <summary>
        /// Verify that the algorithm attribute on ds:SignatureMethod element is set to "http://www.w3.org/2000/09/xmldsig#rsa-sha1".
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_004()
        {
            UploadDocument();
            string algorithm = cdaSignature.GetSignatureMethodAlgorithm();
            LogAssert.AreEqual("http://www.w3.org/2000/09/xmldsig#rsa-sha1", algorithm,
                DialogueResource.SignatureMethodAlgorithm);
        }

        /// <summary>
        /// Verify that the algorithm indicated by the ds:SignatureMethod element has been used to calculate the signature value.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_004_1()
        {
            UploadDocument();
            AsymmetricAlgorithm signingKey = cdaSignature.GetSigningKey();
            string algorithm = cdaSignature.GetSignatureMethodAlgorithm();
            LogAssert.AreEqual(algorithm, signingKey.SignatureAlgorithm,
                DialogueResource.SignatureMethodAlgorithm,
                DialogueResource.AlgorithmUsedToCalculateSignatureValue);
        }

        /// <summary>
        /// Verify that there are one or more ds:Reference elements in ds:SignedInfo element
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_005()
        {
            UploadDocument();
            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInSignedInfo();
            LogAssert.AreEqual(1, referenceElements.Count, DialogueResource.NumberOfReferenceElementsInSignedInfo);
        }

        /// <summary>
        /// Verify that each ds:Reference element in the ds:SignedInfo element contains a URI attribute.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_006()
        {
            UploadDocument();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                LogAssert.IsNotNull(cdaSignature.GetReferenceUri(referenceElement),
                    DialogueResource.UriAttributeOfReferenceElement);
            }
        }

        /// <summary>
        /// Verify that the URI attribute of the ds:Reference element within ds:SignedInfo consists of a "#" character followed by a fragment identifier
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_007()
        {
            UploadDocument();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                string uri = cdaSignature.GetReferenceUri(referenceElement);
                LogAssert.IsTrue(uri.StartsWith("#"),
                    string.Format(DialogueResource.ReferenceUriStartsWithHash, uri),
                    string.Format(DialogueResource.ReferenceUriDoesNotStartWithHash, uri));
                string fragment = uri.Substring(1);
                LogAssert.IsTrue(fragment.Length > 0,
                    string.Format(DialogueResource.ReferenceUriContainsFragment, fragment),
                    DialogueResource.ReferenceUriDoesNotContainFragment);
            }
        }

        /// <summary>
        /// Verify that the fragment identifier listed after the "#" character in the URI attribute matches the id attribute in the sp:signedPayloadData element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_008()
        {
            UploadDocument();
            string signedPayloadDataId = cdaSignature.GetSignedPayloadDataId();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                string uri = cdaSignature.GetReferenceUri(referenceElement);
                string fragment = uri.Substring(1);
                LogAssert.AreEqual(signedPayloadDataId, fragment,
                    DialogueResource.SignedPayloadDataId,
                    DialogueResource.FragmentIdentifierInReferenceUri);
            }
        }

        /// <summary>
        /// Verify that a ds:Transforms element  is present within the ds:Reference element
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_009()
        {
            UploadDocument();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                LogAssert.IsNotNull(cdaSignature.GetTransformsElement(referenceElement),
                    DialogueResource.TransformsElement);
            }
        }

        /// <summary>
        /// Verify that there is only one ds:Transform element nested within the ds:Transforms element
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_010()
        {
            UploadDocument();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                LogAssert.AreEqual(1, cdaSignature.GetTransformElements(referenceElement).Count,
                    DialogueResource.NumberOfTransformElements);
            }
        }

        /// <summary>
        /// Verify that the Exclusive XML Canonicalization method is used on the content being signed.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_011()
        {
            UploadDocument();

            // Load up the CDA signature as a new XML document to sign the signedPayloadData again.
            XmlDocument doc = new XmlDocument();
            doc.Load(new MemoryStream(CdaPackage.ExtractSignature(version.Package)));
            SignedXml xml = new SignedXml(doc);
            Reference reference = new Reference();
            reference.AddTransform(new XmlDsigExcC14NTransform());
            reference.DigestMethod = "http://www.w3.org/2000/09/xmldsig#sha1";
            reference.Uri = "#" + cdaSignature.GetSignedPayloadDataId();
            xml.AddReference(reference);

            // Explicitly set the canonicalisation method to Exclusive XML Canonicalisation
            xml.SignedInfo.CanonicalizationMethod = "http://www.w3.org/2001/10/xml-exc-c14n#";
            xml.SigningKey = patient.GetSigningCertificate().PrivateKey;
            xml.ComputeSignature();

            // Compare the calculated signature value with the actual one in the CDA signature
            string expectedSignatureValue = Convert.ToBase64String(xml.Signature.SignatureValue);
            string actualSignatureValue = cdaSignature.GetSignatureValue();
            LogAssert.AreEqual(expectedSignatureValue, actualSignatureValue,
                DialogueResource.SignatureValueCalculatedUsingExclusiveCanonicalisation,
                DialogueResource.SignatureValueInCdaSignature);
        }

        /// <summary>
        /// Verify that the algorithm attribute in the ds:Transform element has the value of "http://www.w3.org/2001/10/xml-exc-c14n#"
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_012()
        {
            UploadDocument();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                foreach (XmlNode transformElement in cdaSignature.GetTransformElements(referenceElement))
                {
                    LogAssert.AreEqual("http://www.w3.org/2001/10/xml-exc-c14n#",
                        transformElement.Attributes["Algorithm"].Value,
                        DialogueResource.TransformAlgorithm);
                }
            }
        }

        /// <summary>
        /// Verify that the value of the ds:DigestValue element matches the value calculated using the SHA-1 algorithm on the exclusive canonicalization of the signed payload.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_013()
        {
            UploadDocument();
            byte[] excC14nOfSignedPayload = cdaSignature.GetExclusiveCanonicalisationOfSignedPayloadData();
            string sha1HashOfExclusiveC14n = Convert.ToBase64String(SHA1.Create().ComputeHash(excC14nOfSignedPayload));
            XmlNodeList referenceElements = cdaSignature.GetReferenceElementsInSignedInfo();
            string digestValue = cdaSignature.GetReferenceDigestValue(referenceElements[0]);
            LogAssert.AreEqual(sha1HashOfExclusiveC14n, digestValue,
                DialogueResource.HashOfExclusiveXmlCanonicalisation,
                DialogueResource.DigestValueOfSignatureReference);
        }

        /// <summary>
        /// Verify that the algorithm attribute of the ds:DigestMethod element is set to the value of "http://www.w3.org/2000/09/xmldsig#sha1"
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_014()
        {
            UploadDocument();
            foreach (XmlNode referenceElement in cdaSignature.GetReferenceElementsInSignedInfo())
            {
                string digestMethodAlgorithm = cdaSignature.GetDigestMethodAlgorithm(referenceElement);
                LogAssert.AreEqual("http://www.w3.org/2000/09/xmldsig#sha1", digestMethodAlgorithm, DialogueResource.DigestMethodAlgorithm);
            }
        }

        /// <summary>
        /// Verify that the ds:KeyInfo element  is present in the ds:Signature element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_015()
        {
            UploadDocument();
            LogAssert.IsNotNull(cdaSignature.GetKeyInfoElement(), DialogueResource.KeyInfoElementInSignature);
        }

        /// <summary>
        /// Verify that the ds:X509Data element is present in the ds:KeyInfo element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_016()
        {
            UploadDocument();
            LogAssert.IsNotNull(cdaSignature.GetX509DataElement(),
                DialogueResource.X509DataElementInKeyInfo);
        }

        /// <summary>
        /// Verify that the ds:X509Certificate element in the ds:X509Data element is present.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_017()
        {
            UploadDocument();
            LogAssert.IsNotNull(cdaSignature.GetX509CertificateElement(),
                DialogueResource.X509CertificateElementInX509Data);
        }

        /// <summary>
        /// Verify that the ds:X509Certificate element contains the encoded value of the signing certificate.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_018()
        {
            UploadDocument();
            string x509Certificate = cdaSignature.GetX509CertificateElement().InnerText;
            X509Certificate2 cert = new X509Certificate2(Convert.FromBase64String(x509Certificate));
            string actualCertSerial = cert.GetSerialNumberString();

            HospitalDl hospitalDl = new HospitalDl();
            string expectedCertSerial = hospitalDl.Get(patient.TargetHospitalPatient.HospitalId).PcehrCertSerial;

            LogAssert.AreEqual(expectedCertSerial.ToUpper(), actualCertSerial.ToUpper(),
                DialogueResource.SigningCertificateSerialNumber,
                DialogueResource.X509CertificateElementSerialNumber);
        }

        /// <summary>
        /// Verify that the ds:Signature element does NOT contain a ds:Object element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_XSP_SDP_019()
        {
            UploadDocument();
            XmlNode objectElement = cdaSignature.GetObjectElement();
            LogAssert.IsTrue(objectElement == null,
                DialogueResource.SignatureDoesNotContainObjectElement,
                DialogueResource.SignatureContainsObjectElement);
        }

        /// <summary>
        /// Verify that the Signed Payload contains ONLY one ds:Signature element.
        /// </summary>
        [TestMethod]
        [TestCategory("CP_CCA_1A")]
        public void CdaPackaging_1A_PKG_CDA_025()
        {
            UploadDocument();
            XmlNodeList signatureElements = cdaSignature.GetSignatureElements();
            LogAssert.AreEqual(1, signatureElements.Count, DialogueResource.NumberOfSignatureElementsInSignedPayload);
        }
    }
}