﻿using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_204
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems
    ///                                 Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.204
    /// Use Case Name:	Download a Clinical Document
    /// Test Case ID:   PCEHR_CIS_019108
    /// Objective:
    /// To check that the Clinical Information System provides a mechanism to
    /// retrieve:
    /// • A list of clinical documents associated with a healthcare recipient’s
    ///   PCEHR (registryStoredQuery service); and
    /// • A list of historical versions of a particular clinical document from
    ///   the PCEHR System (getChangeHistoryView service).
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019108 : CcaTest
    {
        /// <summary>
        /// Perform an operation to retrieve a list of clinical documents from
        /// the PCEHR System for a healthcare recipient whose PCEHR contains
        /// two or more clinical documents.
        /// a. Verify that the user can access the list of clinical documents
        ///    for the healthcare recipient from the PCEHR System.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_204")]
        public void CIS_204_019108_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");

            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // Get the list of documents
            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListActive);

            // Check that the request was audited (no separate test for this)
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetDocumentList, audit.ServiceName, DialogueResource.PcehrAuditServiceName);
        }

        /// <summary>
        /// Perform an operation to retrieve a list of historical versions from
        /// the PCEHR System for a healthcare recipient whose PCEHR contains a
        /// clinical document with historical versions.
        /// b. Verify that the user can access the list of historical versions
        ///    of the clinical document for the healthcare recipient from the
        ///    PCEHR System.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_204")]
        public void CIS_204_019108_B()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");
            UserDetails user = patient.GetTestUser();
            PatientIdentifierBase identifier = patient.TargetPatientIdentifier;
            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(identifier, user);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            //DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentListActive(identifier, user);
            //Assert.AreEqual(HipsResponseIndicator.OK, listResponse.HipsResponse.Status);
            //DocumentMetaDataItem item = listResponse.DocumentList.First();

            // This is Test Document ID 106 in the Conformance Test Data for CIS connecting to PCEHR v1.8
            string documentId = CdaDocument.ConvertToOid("ebf96c6f-3252-47b0-aa89-b72740d5672d");

            // Get the list of historical versions and verify that there are 4 versions and all are event summaries
            ChangeHistoryViewResponse<PatientIdentifierBase> historyResponse = ProxyHelper.PcehrProxy.GetChangeHistoryView(identifier, user, documentId);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, historyResponse.HipsResponse, DialogueResource.HipsServiceGetChangeHistoryView);
            LogAssert.AreEqual(4, historyResponse.DocumentList.Count, DialogueResource.DocumentListCount);
            foreach (DocumentMetaDataItem item in historyResponse.DocumentList)
            {
                LogAssert.AreEqual("Event Summary", historyResponse.DocumentList[0].DocumentClassName, DialogueResource.DocumentClassName);
            }

            // Check that the request was audited (no separate test for this)
            PcehrAudit audit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, audit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
            LogAssert.AreEqual(AuditOperationNames.GetChangeHistoryView, audit.ServiceName, DialogueResource.PcehrAuditServiceName);
        }
    }
}