﻿using System;
using System.Linq;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nehta.VendorLibrary.PCEHR;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_204
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems
    ///                                 Connecting to the PCEHR System
    ///                                 Version 1.6 — 27 September 2012
    /// Use Case #:	    UC.CIS.204
    /// Use Case Name:	Download a Clinical Document
    /// Test Case ID:   PCEHR_CIS_019118
    /// Objective:
    /// To check that the Clinical Information System provides an ability to
    /// sort and filter the lists of clinical documents retrieved from the
    /// PCEHR System.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_019118 : CcaTest
    {
        /// <summary>
        /// Perform an operation to retrieve a list of clinical documents from
        /// the PCEHR System for a healthcare recipient whose PCEHR contains
        /// two or more clinical documents.
        /// a. Verify that the user can sort and filter the list of clinical
        ///    documents for the healthcare recipient.
        ///
        /// Partial Compliance: HIPS can filter but does not sort the list.
        /// </summary>
        [TestMethod]
        [TestCategory("CIS_CCA_204")]
        public void CIS_204_019118_A()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "CCA ID IHI_87");

            GainPcehrAccessResponse accessResponse = ProxyHelper.PcehrProxy.GainAccessEmergency(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, accessResponse.HipsResponse, DialogueResource.HipsServiceGainAccessEmergency);

            // Get the list of documents with a filter - here demonstrating the creation time and document type filters
            DocumentQuery docQuery = new DocumentQuery();
            docQuery.CreationTimeFrom = DateTime.Parse("01/01/2012");
            docQuery.CreationTimeTo = DateTime.Parse("01/01/2013");
            docQuery.DocumentClassCode = new ClassCodes[] { ClassCodes.DischargeSummary };
            string[] expectedClassCodes = (from code in docQuery.DocumentClassCode
                                           select code.GetAttributeValue<CodedValueAttribute, string>(a => a.ConceptCode)).ToArray();

            DocumentListResponse<PatientIdentifierBase> listResponse = ProxyHelper.PcehrProxy.GetDocumentList(patient.TargetPatientIdentifier, patient.GetTestUser(), docQuery);
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, listResponse.HipsResponse, DialogueResource.HipsServiceGetDocumentListWithQuery);
            if (listResponse.DocumentList.Count == 0)
            {
                LogAssert.Inconclusive(DialogueResource.CIS_204_019118_A_NoDocs);
            }
            foreach (DocumentMetaDataItem item in listResponse.DocumentList)
            {
                // Check that every list item has a creation time within the filtered range
                LogAssert.IsBetween(docQuery.CreationTimeFrom, docQuery.CreationTimeTo, item.CreationTime, DialogueResource.DocumentCreationTime);

                // Check that every list item is one of the filtered document types
                LogAssert.Contains(expectedClassCodes, item.DocumentClassCode, DialogueResource.DocumentClassCode);
            }
            DocumentListResponse<PatientIdentifierBase> unfiltered = ProxyHelper.PcehrProxy.GetDocumentList(patient.TargetPatientIdentifier, patient.GetTestUser());
            LogAssert.ExpectResponse(HipsResponseIndicator.OK, unfiltered.HipsResponse, DialogueResource.HipsServiceGetDocumentListUnfiltered);
            if (unfiltered.DocumentList.Count == listResponse.DocumentList.Count)
            {
                LogAssert.Inconclusive(DialogueResource.CIS_204_019118_A_SameCount);
            }
        }
    }
}