﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_501
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.501
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020208f
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020208f : CcaTest
    {
        /// <summary>
        /// Prepare registration details without an IVC required (i.e. IVC delivery method of 'None'), and complete all other necessary fields for a successful registration.
        ///     a. Submit the registration request.
        ///     b. Verify the CIS requires the user to specify the IVC delivery method or defaults to a delivery method of 'None'.
        ///     c. Verify the registration request sent to the eHealth record system contains an IVC delivery method of 'None'.
        ///
        /// Note: The ability for the CIS to support IVC delivery methods such as "SMS", "Email" and "Response" are covered by PCEHR Notice Of Connection (NOC) Test Case #95, 102, 103 and CCA Test Case PCEHR_CIS_020205, respectively.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020208f()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 05");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel and set to IVC channel of None
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.none;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that HIPS audited the request
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyRegisterPatientIvcChannelNone(pcehrAudit);

        }

        private void VerifyRegisterPatientIvcChannelNone(PcehrAudit audit)
        {
            // Confirm that the payload is well formed and complies with the Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");

            // Check that the IVC Channel was set to None
            XmlNode ivcChannelNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:assertions/p:ivcCorrespondence/p:channel", nsmgr);
            LogAssert.AreEqual(ivcChannelNode.InnerText, "none", "IVC channel is not None");
        }
    }
}
