﻿using System.Collections.Generic;
using HIPS.CommonSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_501
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 Amendment 1 ——— 7 March 2013
    /// Use Case #:	    UC.CIS.501
    /// Use Case Name:	Assisted Registration of an adult
    /// Purpose:        To assist a consumer with registering for an eHealth record.
    /// 
    /// Test Case ID:   PCEHR_CIS_020209
    /// Objective:      To verify the CIS ensures the IHI is valid before
    ///                 including it in the registration request.
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020209 : CcaTest
    {
        /// <summary>
        /// If the CIS supports the capability to provide a consumer's IHI in
        /// the registration request, then: 
        /// 
        /// 1. Prepare registration details that include an IHI that is not
        ///    correct for a consumer, and complete all necessary registration
        ///    fields.
        ///    
        ///    a. Ensure the CIS validates the IHI against the HI Service, and
        ///       that the registration request is not sent to the eHealth
        ///       record system.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020209_1()
        {
            patient = CcaPatient.GetPatient(ihiValid: false, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 08");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel and set to IVC channel of response
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            // After validating the IHI it should have an unresolved alert (Demographic Mismatch).
            LogAssert.ExpectResponse(HipsResponseIndicator.UnresolvedIhiAlert, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that HIPS did attempt to validate the IHI for this patient
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did not access the PCEHR
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreNotEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }

        /// <summary>
        /// If the CIS supports the capability to provide a consumer's IHI in
        /// the registration request, then: 
        /// 
        /// 2. Prepare registration details that include an IHI that is correct
        ///    for a consumer, and complete all necessary registration fields.
        ///    
        ///    a. Ensure that the CIS validates the IHI against the HI Service
        ///       before sending the registration request to the eHealth record
        ///       system.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_501")]
        public void CIS_501_020209_2()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 01");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel and set to IVC channel of response
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterPatient(patient.TargetPatientIdentifier,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                patient.GetTestUser());

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that HIPS did attempt to validate the IHI for this patient
            IhiLookupAudit hiAudit = patient.GetLastIhiLookupAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, hiAudit.PatientMasterId, DialogueResource.IhiLookupAuditPatientMasterId);

            // Check that HIPS did access the PCEHR to register the patient
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit();
            LogAssert.AreEqual(patient.TargetPatientMasterId, pcehrAudit.PatientMasterId, DialogueResource.PcehrAuditPatientMasterId);
        }
    }
}
