﻿using System;
using System.Text;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.CommonCcaNoc.Helpers;
using HIPS.PcehrSchemas;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using System.Xml;
using System.IO;

namespace Test.PcehrCcaNoc.ClinicalInformationSystem.UC_CIS_502
{
    /// <summary>
    /// Conformance Test Specification: Clinical Information Systems Connecting to the PCEHR System
    ///                                 Version 1.6 — 7 March 2013
    /// Use Case #:	    UC.CIS.502
    /// Use Case Name:	Assisted Registration of an adult
    /// Test Case ID:   PCEHR_CIS_020208h
    /// Objective:
    /// To assist a consumer with registering for an eHealth record. 
    /// </summary>
    [TestClass]
    public class PCEHR_CIS_020208h : CcaTest
    {
        /// <summary>
        /// If the clinical information system is capable of sending an electronic copy of the signed assisted registration form to the eHealth record system along with the registration request, then:
        ///     1. Prepare registration details that include an electronic copy of a paper-based registration form.
        ///         a. Submit the registration request
        ///         b. Verify the request sent to the eHealth record system contains the electronic copy of the paper-based registration form.
        /// </summary>
        [TestMethod]
        [TestCategory("Assisted Registration CIS_CCA_502")]
        public void CIS_502_020208h()
        {
            patient = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 21");

            // Representative with incorrect demographics
            CcaPatient patientRep = CcaPatient.GetPatient(ihiValid: true, validatedWithinPeriod: true, hasUnresolvedAlerts: false, testDataId: "AR ID 19");

            // Create document consent list
            List<RegistrationDocumentConsent> documentConsentList = new List<RegistrationDocumentConsent>() { 
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.ACIR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.AODR, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBS, RegistrationDocumentConsentStatus.ConsentGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.MBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBS, RegistrationDocumentConsentStatus.ConsentNotGiven),
                new RegistrationDocumentConsent(RegistrationDocumentConsentType.PBSPastAssimilation, RegistrationDocumentConsentStatus.ConsentNotGiven),
            };

            // Create correspondence channel and set to IVC channel of response
            RegistrationCorrespondenceChannel channel = new RegistrationCorrespondenceChannel();
            channel.Channel = IvcCorrespondenceChannel.response;

            // Add the consent form
            Attachment consentForm = new Attachment();

            FileInfo file = new FileInfo(CcaPatient.TestDocumentPath + @"UC.CIS.502.20208h.png");

            consentForm.FileName = file.Name;
            consentForm.Contents = File.ReadAllBytes(file.FullName);

            PcehrRegistrationResponse registrationResponse = ProxyHelper.AssistedRegistrationProxy.RegisterDependant(patient.TargetPatientIdentifier,
                patientRep.TargetDemographic,
                EvidenceOfIdentityType.IdentityVerificationMethod1,
                IndigenousStatus.NeitherAboriginalAndTorresStraitIslander,
                documentConsentList,
                channel,
                true,
                true,
                patient.GetTestUser(),
                consentForm);

            LogAssert.ExpectResponse(HipsResponseIndicator.OK, registrationResponse.HipsResponse, DialogueResource.HipsServiceRegisterPatient);

            // Check that HIPS audited the request
            PcehrAudit pcehrAudit = patient.GetLastPcehrAudit("Register PCEHR");

            VerifyRegisterPatientConsentForm(pcehrAudit);

        }

        private void VerifyRegisterPatientConsentForm(PcehrAudit audit)
        {
            // Confirm that the payload is well formed and complies with the Request XML Schema
            XmlDocument doc = new XmlDocument();
            using (StringReader sr = new StringReader(audit.Request))
            {
                doc.Load(sr);
            }
            XmlNamespaceManager nsmgr = new XmlNamespaceManager(new NameTable());
            nsmgr.AddNamespace("s", "http://www.w3.org/2003/05/soap-envelope");
            nsmgr.AddNamespace("p", "http://ns.electronichealth.net.au/pcehr/xsd/interfaces/RegisterPCEHR/2.0");
            nsmgr.AddNamespace("c", "http://ns.electronichealth.net.au/pcehr/xsd/common/CommonCoreElements/1.0");

            // Check that the consent form has data
            XmlNode consentFormNode = doc.SelectSingleNode("/s:Envelope/s:Body/p:registerPCEHR/p:assertions/p:identity/p:signedConsentForm", nsmgr);
            LogAssert.IsNotNull(consentFormNode, "Consent form is blank");
        }
    }
}
