﻿using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PatientSchemas;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.ConsentManagement;
using HIPS.Web.ModelInterface.DischargeSummary;
using HIPS.Web.ModelInterface.DocumentManagement;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.DocumentUpload;
using HIPS.Web.UI.ViewModels.Shared;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Web.Mvc;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for discharge summary upload feature.
    /// </summary>
    [HpoRequired]
    public class DocumentUploadController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Patient repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientRepository;

        /// <summary>
        /// Cda repository to be used by this controller.
        /// </summary>
        private readonly ICdaRepository cdaRepository;

        /// <summary>
        /// Episode repository to be used by this controller.
        /// </summary>
        private readonly IEpisodeRepository episodeRepository;

        /// <summary>
        /// Episode V2 repository to be used by this controller.
        /// </summary>
        private readonly IEpisodeRepositoryV2 episodeRepositoryV2;

        /// <summary>
        /// Uploaded document repository to be used by this controller.
        /// </summary>
        private readonly IUploadedDocumentRepository uploadedDocumentRepository;

        /// <summary>
        /// Discharge summary reference repository to be used by this controller.
        /// </summary>
        private readonly IDischargeSummaryReferenceRepository dischargeSummaryRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="DocumentUploadController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="episodeRepository">Episode repository to be used by this controller.</param>
        /// <param name="episodeRepositoryV2">Episode V2 repository to be used by this controller.</param>
        /// <param name="cdaRepository">Cda repository to be used by this controller.</param>
        /// <param name="uploadedDocumentRepository">Uploaded document repository to be used by this controller.</param>
        /// <param name="dischargeSummaryRepository">Discharge summary reference repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public DocumentUploadController(
            IHospitalRepository hospitalRepository,
            IPatientRepository patientRepository,
            IEpisodeRepository episodeRepository,
            IEpisodeRepositoryV2 episodeRepositoryV2,
            ICdaRepository cdaRepository,
            IUploadedDocumentRepository uploadedDocumentRepository,
            IDischargeSummaryReferenceRepository dischargeSummaryRepository,
            ISettingsRepository settingsRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.patientRepository = patientRepository;

            this.episodeRepository = episodeRepository;
            this.episodeRepositoryV2 = episodeRepositoryV2;

            this.cdaRepository = cdaRepository;
            this.uploadedDocumentRepository = uploadedDocumentRepository;
            this.dischargeSummaryRepository = dischargeSummaryRepository;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets the number of days after discharge to still show an episode on
        /// the list of episodes for which the patient can upload documents to their PCEHR.
        /// </summary>
        private int DischargeSummaryPatientListDaysDischarged
        {
            get
            {
                string setting = this.Settings.GetSettingValue(Setting.SettingCodes.DischargeSummaryPatientListDaysDischarged);
                return int.Parse(setting);
            }
        }

        /// <summary>
        /// Gets the discharge summary document type code.
        /// </summary>
        private string DischargeSummaryDocumentTypeCode
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.DischargeSummaryDocumentTypeCode);
            }
        }

        /// <summary>
        /// Gets the discharge summary document format code for uploading to the patient's PCEHR.
        /// </summary>
        private string DischargeSummaryDocumentFormatCode
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.DischargeSummaryDocumentFormatCode);
            }
        }

        /// <summary>
        /// Gets the discharge summary document author.
        /// </summary>
        private string DischargeSummaryDocumentAuthor
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.DischargeSummaryAuthor);
            }
        }

        /// <summary>
        /// Gets the discharge summary document responsible health professional (RHP).
        /// </summary>
        private string DischargeSummaryDocumentRHP
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.DischargeSummaryRHP);
            }
        }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Display a list of patients for a selected hospital.
        /// </summary>
        /// <param name="lookupMessages">Additional messages to be displayed coming from the lookup MRN action.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Patients(ViewMessageList lookupMessages = null)
        {
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            var model = new PatientsViewModel() { HospitalId = hospitalId };
            this.LoadCurrentContext(model);

            // Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                // Update ViewModel with reference data:
                model.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                model.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            if (hospitalId != null)
            {
                var response = this.patientRepository.ListPatientsCurrentlyInHospital(this.DefaultHospitalCodeSystem, hospitalId, null, this.GetCurrentUserDetails());

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    if (response.Data.PatientInHospitalList.Count() > 0)
                    {
                        // Update ViewModel with patients.
                        model.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(response.Data.PatientInHospitalList));
                    }
                    else
                    {
                        model.Messages.Add("There are no current patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";

                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));

                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    model.Messages.Add(errorMessage, MessageLevel.Error);
                }

                if (lookupMessages != null)
                {
                    model.Messages.AddRange(lookupMessages);
                }
            }

            return this.View("Patients", model);
        }

        /// <summary>
        /// Look up a patient by MRN. If found then display the list of
        /// episodes for that patient, otherwise redisplay the list of patients
        /// with a message indicating that the patient was not found.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="lookupMrn">User-entered identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult LookupMrn(string hospitalId, string lookupMrn)
        {
            var model = new PatientsViewModel { HospitalId = hospitalId, PatientId = lookupMrn };
            this.LoadCurrentContext(model);

            if (model.CurrentPatient != null)
            {
                return this.Episodes(hospitalId, lookupMrn);
            }
            else
            {
                ViewMessageList messages = new ViewMessageList();
                messages.Add("There is no current patient with this MRN.", MessageLevel.Error);

                return this.Patients(messages);
            }
        }

        /// <summary>
        /// Display a list of episodes for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Episodes(string hospitalId, string patientId)
        {
            var model = new EpisodesViewModel() { PatientId = patientId, HospitalId = hospitalId };
            this.LoadCurrentContext(model);

            var patientIdentifier = new Mrn(model.PatientId, model.HospitalId, model.CurrentHospital.CodeSystemCode);
            var response = this.episodeRepositoryV2.ListPatientEpisodesInHospital(patientIdentifier, true, this.DischargeSummaryDocumentTypeCode, this.GetLocalUser());

            if (response.IsSuccessful)
            {
                if (response.Data != null)
                {
                    if (response.Data.Count > 0)
                    {
                        var episodes = from episode in response.Data
                                       from document in episode.LocalClinicalDocumentMetaData.DefaultIfEmpty()
                                       select new EpisodeViewModel
                                       {
                                           AdmissionDate = document == null ? episode.AdmissionDate : document.AdmissionDate,
                                           DischargeDate = document == null ? episode.DischargeDate : document.DischargeDate,
                                           AdmissionReason = document == null ? episode.AdmissionReason : document.AdmissionReason,
                                           ResponsibleProviderFamilyName = document == null ? episode.ResponsibleProviderFamilyName : document.ResponsibleProviderFamilyName,
                                           ResponsibleProviderGivenNames = document == null ? episode.ResponsibleProviderGivenNames : document.ResponsibleProviderGivenNames,
                                           ResponsibleProviderSuffix = document == null ? episode.ResponsibleProviderSuffix : document.ResponsibleProviderSuffix,
                                           ResponsibleProviderTitle = document == null ? episode.ResponsibleProviderTitle : document.ResponsibleProviderTitle,
                                           Status = document == null ? HIPS.Web.Model.LocalDocumentStatus.Unknown : document.Status,
                                           ConsentWithdrawn = episode.ConsentWithdrawn.Value,
                                           HasPcehr = model.CurrentPatient.HasPcehr == "Yes" ? true : false,
                                           SourceSystemEpisodeId = episode.SourceSystemEpisodeId,
                                           SourceSystemSetId = document == null ? null : document.SourceSystemSetId
                                       };

                        model.Episodes.AddRange(episodes);
                    }
                    else
                    {
                        model.Messages.Add(string.Format("The selected patient has no episodes within the past {0} days.", this.DischargeSummaryPatientListDaysDischarged), MessageLevel.Information);
                    }
                }
                else
                {
                    model.Messages.AddRange(ObjectMapper.Map<ViewMessageList>(response.Messages));
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve episodes for the selected patient.";

                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));

                // Display error message.
                this.SetAjaxErrorResponseCode();
                model.Messages.Add(errorMessage, MessageLevel.Error);
            }

            return this.View("Episodes", model);
        }

        /// <summary>
        /// Display a list of episodes for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient (MRN).</param>
        /// <param name="statePatientId">Identifier of the patient (State Patient Id).</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult EmbeddedEnterpriseEpisodes(string hospitalId, string patientId, string statePatientId)
        {
            if (string.IsNullOrWhiteSpace(statePatientId))
            {
                statePatientId = patientId;
            }

            var model = new EpisodesViewModel() { PatientId = patientId, StatePatientId = statePatientId, HospitalId = hospitalId };
            this.LoadCurrentContext(model);

            var patientIdentifier = new RegisteredEnterprisePatient(model.PatientId, statePatientId, model.CurrentHospital.Code, model.CurrentHospital.CodeSystemCode);
            var response = this.episodeRepositoryV2.ListPatientEpisodesInHospital(patientIdentifier, true, this.DischargeSummaryDocumentTypeCode, this.GetLocalUser());

            if (response.IsSuccessful)
            {
                if (response.Data.Count > 0)
                {
                    var episodes = from episode in response.Data
                                   from document in episode.LocalClinicalDocumentMetaData.DefaultIfEmpty()
                                   select new EpisodeViewModel
                                   {
                                       AdmissionDate = document == null ? episode.AdmissionDate : document.AdmissionDate,
                                       DischargeDate = document == null ? episode.DischargeDate : document.DischargeDate,
                                       AdmissionReason = document == null ? episode.AdmissionReason : document.AdmissionReason,
                                       ResponsibleProviderFamilyName = document == null ? episode.ResponsibleProviderFamilyName : document.ResponsibleProviderFamilyName,
                                       ResponsibleProviderGivenNames = document == null ? episode.ResponsibleProviderGivenNames : document.ResponsibleProviderGivenNames,
                                       ResponsibleProviderSuffix = document == null ? episode.ResponsibleProviderSuffix : document.ResponsibleProviderSuffix,
                                       ResponsibleProviderTitle = document == null ? episode.ResponsibleProviderTitle : document.ResponsibleProviderTitle,
                                       Status = document == null ? HIPS.Web.Model.LocalDocumentStatus.Unknown : document.Status,
                                       ConsentWithdrawn = episode.ConsentWithdrawn.Value,
                                       HasPcehr = model.CurrentPatient.HasPcehr == "Yes" ? true : false,
                                       SourceSystemEpisodeId = episode.SourceSystemEpisodeId,
                                       SourceSystemSetId = document == null ? null : document.SourceSystemSetId
                                   };

                    model.Episodes.AddRange(episodes);
                }
                else
                {
                    model.Messages.Add(string.Format("The selected patient has no episodes within the past {0} days.", this.DischargeSummaryPatientListDaysDischarged), MessageLevel.Information);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve episodes for the selected patient.";

                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));

                // Display error message.
                this.SetAjaxErrorResponseCode();
                model.Messages.Add(errorMessage, MessageLevel.Error);
            }

            return this.View("Episodes", model);
        }

        /// <summary>
        /// Display a form for uploading/superseding discharge summary document.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="episodeId">Identifier of the episode.</param>
        /// <param name="setId">Identifier of the document.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult UploadOrSupersede(string hospitalId, string patientId, string statePatientId, string episodeId, string setId)
        {
            if (string.IsNullOrWhiteSpace(statePatientId))
            {
                statePatientId = patientId;
            }
            
            var model = new CreateDocumentViewModel() { HospitalId = hospitalId, PatientId = patientId, StatePatientId = statePatientId, EpisodeId = episodeId, SetId = setId };
            this.LoadCurrentContext(model);

            if (model.CurrentPatient == null)
            {
                model.Messages.Add(string.Format("Patient {0} not found.", patientId), MessageLevel.Information);
                return this.View("DocumentUpload", model);
            }

            model.ModesOfSeparation = Enum.GetValues(typeof(HIPS.Web.Model.ModeOfSeparation))
                                             .Cast<HIPS.Web.Model.ModeOfSeparation>()
                                             .Where(m => m != HIPS.Web.Model.ModeOfSeparation.None)
                                             .ToSelectListItems(m => ((int)m).ToString(), m => model.SetModeOfSeparationDescription(m));

            model.SourceDocumentStatus = Enum.GetValues(typeof(HIPS.Web.Model.SourceDocumentStatus))
                                             .Cast<HIPS.Web.Model.SourceDocumentStatus>()
                                             .Where(s => s != HIPS.Web.Model.SourceDocumentStatus.None)
                                             .ToSelectListItems(s => ((int)s).ToString(), s => s.ToString());

            model.ClinicalSpecialty = this.dischargeSummaryRepository.GetClinicalSpecialties().ToSelectListItems(c => c.Code, c => c.Description);

            model.CurrentUser = new UserViewModel();
            model.ResponsibleUser = new UserViewModel();

            // Load current user.
            this.SetUser(model.CurrentUser, model.CurrentEpisode, this.DischargeSummaryDocumentAuthor, this.DischargeSummaryDocumentFormatCode);

            // Load current responsible user.
            this.SetUser(model.ResponsibleUser, model.CurrentEpisode, this.DischargeSummaryDocumentRHP, this.DischargeSummaryDocumentFormatCode);

            return this.View("DocumentUpload", model);
        }

        /// <summary>
        /// Process the AJAX request to upload/supersede discharge summary document.
        /// </summary>
        /// <param name="model">Model that contains data to be processed.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult UploadOrSupersede(UploadDocumentViewModel model)
        {
            var document = this.uploadedDocumentRepository.GetCdaDocumentDetails(model.DocumentSetId);
            var response = this.uploadedDocumentRepository.UploadOrSupersedeDocument(
                (HIPS.Web.Model.PatientIdentifier.Mrn)document.PatientIdentifier,
                document.Document.Content,
                document.Document.Attachments,
                model.AdmissionDate,
                this.DischargeSummaryDocumentFormatCode,
                this.GetCurrentUserDetails());

            return this.Json(new { Response = response, Errors = ModelState.ToErrorDictionary() });
        }

        /// <summary>
        /// Process the AJAX request for previewing discharge summary document. The result
        /// is a partial view to be loaded on a modal.
        /// </summary>
        /// <param name="model">Model that contains data to be processed.</param>
        /// <returns>The partial view result.</returns>
        [HttpPost]
        [ValidateAntiForgeryToken]
        public ActionResult ViewDocument(CreateDocumentViewModel model)
        {
            var messages = new ViewMessageList();

            if (!this.ModelState.IsValid)
            {
                var errors = this.ModelState.Values.SelectMany(x => x.Errors);
                
                foreach (var error in errors)
                {
                    messages.Add(error.ErrorMessage, MessageLevel.Error);
                }

                return this.PartialView("ViewMessageList", messages);
            }

            byte[] pdfDocument = null;
            using (var reader = new BinaryReader(model.DischargeSummaryAttachment.InputStream))
            {
                pdfDocument = reader.ReadBytes(model.DischargeSummaryAttachment.ContentLength);
            }

            // image attachments
            var cdaAttachments = new List<HIPS.Web.Model.Cda.CdaAttachment>();
            foreach (var attachment in model.ImageAttachments)
            {
                if (attachment != null && attachment.ContentLength != 0)
                {
                    var cdaAttachment = new HIPS.Web.Model.Cda.CdaAttachment();
                    cdaAttachment.FileName = Path.GetFileName(attachment.FileName);

                    using (var reader = new BinaryReader(attachment.InputStream))
                    {
                        cdaAttachment.Content = reader.ReadBytes(attachment.ContentLength);

                        if (cdaAttachment.AttachmentType == Model.AttachmentType.OrganisationalLogoPng) 
                        {
                            using (var image = Bitmap.FromStream(attachment.InputStream))
                            {
                                if (image.Height > 100 || image.Width > 400)
                                {
                                    string message = string.Format("Organisational logo with size {0}x{1} pixels exceeds the 400x100 limit for uploading to PCEHR.", image.Width, image.Height);
                                    messages.Add(message, MessageLevel.Error);

                                    return this.PartialView("ViewMessageList", messages);
                                }
                            }
                        }
                    }

                    cdaAttachments.Add(cdaAttachment);
                }
            }

            var metadata = ObjectMapper.Map<HIPS.Web.Model.Cda.CdaHeaderMetadata>(model);

            this.LoadCurrentContext(model);
            var patientIdentifier = new HIPS.Web.Model.PatientIdentifier.Mrn() { Value = model.PatientId, HospitalCode = model.HospitalId, HospitalCodeSystem = model.CurrentHospital.CodeSystemCode };

            var request = this.cdaRepository.CreateDischargeSummaryLevel1A(patientIdentifier, metadata, cdaAttachments, pdfDocument, this.GetLocalUser());
            messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));

            if (request.IsSuccessful)
            {
                this.uploadedDocumentRepository.SetCdaDocumentDetails(request.Data.DocumentSetId, request.Data);

                var result = new UploadDocumentViewModel()
                {
                    HospitalId = model.HospitalId,
                    PatientId = model.PatientId,
                    DocumentSetId = request.Data.DocumentSetId,
                    AdmissionDate = model.CurrentEpisode.AdmissionDate,
                    DocumentStatus = model.CurrentEpisode.DocumentStatus
                };

                return this.PartialView("ViewDocument", result);
            }
            else
            {
                string errorMessage = "Failed to upload discharge summary for the selected patient.";

                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new Exception(string.Format("{0} {1}", errorMessage, request.Messages.AsString())));

                // Display error message.
                this.SetAjaxErrorResponseCode();

                messages.AddRange(ObjectMapper.Map<ViewMessageList>(request.Messages));
                
                return this.PartialView("ViewMessageList", messages);
            }
        }

        /// <summary>
        /// Display the selected document. The result is a partial view to
        /// be loaded into the container on the ViewDocument page. The view
        /// contains an informational message regarding the source of the
        /// document, and an i-frame in which the rendered HTML will be shown.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="setId">Identifier of the document.</param>
        /// <returns>The partial view result.</returns>
        [HttpGet]
        public ActionResult DocumentContent(string hospitalId, string patientId, string setId)
        {
            // Retrieve document
            var document = this.uploadedDocumentRepository.GetCdaDocumentDetails(setId);
            var model = new DocumentContentViewModel();
            model.DocumentUrl = Url.Action("DocumentContentView", new { hospitalId, patientId, setId, filename = "CDA_ROOT.html" });

            return this.PartialView(model);
        }

        /// <summary>
        /// Gets a file from a document package. For the file name "CDA_ROOT.html"
        /// it will return the HTML that is produced by applying the NEHTA
        /// style sheet to the CDA document. For the file name "CDA_ROOT.xml" it
        /// will return the raw XML source, which is not required for the web
        /// site but is useful for testing. For other file names, it will return
        /// the attachment file.
        /// </summary>
        /// <param name="hospitalId">Identifier of the selected hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="setId">Identifier of the selected document.</param>
        /// <param name="filename">Name of the file that is requested.</param>
        /// <returns>HTML Content, File or FileInline.</returns>
        [HttpGet]
        public ActionResult DocumentContentView(string hospitalId, string patientId, string setId, string filename)
        {
            // Check if request is for a filename that the filename is valid
            if (!string.IsNullOrWhiteSpace(filename))
            {
                // If filename is missing an extension, or has a directory, then it is invalid
                bool hasDirectory = !string.IsNullOrWhiteSpace(Path.GetDirectoryName(filename));
                if (!Path.HasExtension(filename) || hasDirectory)
                {
                    return this.HttpNotFound();
                }
            }

            // Retrieve document
            var document = this.uploadedDocumentRepository.GetCdaDocumentDetails(setId);

            // Looking for rendered HTML (no filename or filename is "CDA_ROOT.html")
            if (filename == null || filename.Equals("CDA_ROOT.html", StringComparison.OrdinalIgnoreCase))
            {
                return this.Content(document.Document.Content.TransformToHtml(Properties.Resources.NEHTA_Generic_CDA_Stylesheet_1_2_9));
            }

            // Looking for XML source
            if (filename.Equals("CDA_ROOT.xml", StringComparison.OrdinalIgnoreCase))
            {
                return this.File(document.Document.Content, "text/xml");
            }

            // Looking for an attachment
            {
                var attachment = document.Document.Attachments.FirstOrDefault(a => a.FileName.Equals(filename, StringComparison.OrdinalIgnoreCase));

                // If nothing found, return 404
                if (attachment == null)
                {
                    return this.HttpNotFound();
                }

                // Return attachment with filename and inferred MIME type, attempt to display inline rather than prompt download
                return this.FileInlineInferMime(attachment.Content, filename);
            }
        }

        #region Helpers

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        private void LoadCurrentContext(DocumentUploadViewModelBase model)
        {
            var mrn = new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem);
            var user = this.GetCurrentUserDetails();

            // Load current hospital.
            if (!string.IsNullOrEmpty(model.HospitalId))
            {
                var response = this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem);
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(response, new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);
                }
            }

            // Load current patient.
            if (!string.IsNullOrEmpty(model.PatientId))
            {
                var isEmbeddedEnterprisePage = this.HttpContext.Request.Url.AbsolutePath.StartsWith("/EmbeddedEnterprise");
                ServiceResponse<PatientDisclosureDetailsResponse> response;

                if(!isEmbeddedEnterprisePage)
                {
                    response = this.patientRepository.GetPatientDisclosureDetails(user, mrn);                    
                }
                else
                {
                    var patientIdentifier = new RegisteredEnterprisePatient(model.PatientId, model.StatePatientId, model.HospitalId, this.DefaultHospitalCodeSystem);
                    response = this.patientRepository.GetPatientDisclosureDetails(user, patientIdentifier);                    
                }
                
                model.CurrentPatient = ObjectMapper.Map<PatientViewModel>(response.Data.AdmittedPatient);
            }

            // Load current episode.
            if (!string.IsNullOrEmpty(model.EpisodeId))
            {
                var response = this.episodeRepository.GetEpisodeDetails(user, model.EpisodeId, mrn);
                model.CurrentEpisode = ObjectMapper.Map<EpisodeViewModel>(response.Data);
            }

            if (!string.IsNullOrEmpty(model.SetId))
            {
                var response = this.uploadedDocumentRepository.Get(user, model.SetId, null, mrn);
                model.CurrentEpisode.Status = (HIPS.Web.Model.LocalDocumentStatus)response.Data.LocalClinicalDocumentMetaData.Status;
            }
            
        }

        /// <summary>
        /// Set user depending on author setting.
        /// </summary>
        /// <param name="model">User model.</param>
        /// <param name="episode">Episode model.</param>
        /// <param name="setting">User Setting.</param>
        private void SetUser(UserViewModel model, EpisodeViewModel episode, string setting, string documentFormatCodeSetting)
        {
            if (setting == "Episode")
            {
                model.UserID = episode.ResponsibleProviderIdentifier;
                model.FamilyName = episode.ResponsibleProviderFamilyName;
                model.GivenNames = episode.ResponsibleProviderGivenNames;
            }
            else if (setting == "User")
            {
                var user = this.GetLocalUser();
                model.UserID = user.Login;
                model.FamilyName = user.FamilyName;
                model.GivenNames = user.GivenNames;
            }
            model.IsHpiiEnforcedDocumentFormat = (documentFormatCodeSetting == HIPS.CommonSchemas.Cda.Constants.DocumentFormatCode.DISCHARGE_SUMMARY_1A_ENFORCED);
        }

        #endregion Helpers

        #endregion Methods
    }
}