﻿#region Using Statements

using System;
using System.Collections.Generic;
using System.Web.Mvc;
using System.Web.Routing;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.ViewModels.PcehrView;

#endregion Using Statements

namespace HIPS.Web.UI.Filters
{
    /// <summary>
    /// Provides an action filter that supports checking access to the PCEHR.
    /// </summary>
    public class PcehrAccessFilter : IActionFilter
    {
        #region Fields

        /// <summary>
        /// List of application-specific settings.
        /// </summary>
        private readonly List<Setting> Settings;

        /// <summary>
        /// Gets the patient repository to be used by this class.
        /// </summary>
        private readonly IPatientRepository PatientRepository;

        /// <summary>
        /// Gets the PCEHR view repository to be used by this class.
        /// </summary>
        private readonly IPcehrViewRepository PcehrViewRepository;

        private readonly IPcehrServiceV2Repository PcehrServiceV2Repository;

        #endregion Fields

        #region Properties

        /// <summary>
        /// Gets the default hospital code system used by the application.
        /// </summary>
        private string DefaultHospitalCodeSystem
        {
            get
            {
                return Settings.GetSettingValue(Setting.SettingCodes.DefaultHospitalCodeSystem);
            }
        }

        /// <summary>
        /// Gets or sets a Boolean value that indicates whether to automatically attempt to gain access if required.
        /// </summary>
        private bool AutoAccess { get; set; }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="patientRepository">Patient repository to be used by this class.</param>
        /// <param name="pcehrViewRepository">PCEHR view repository to be used by this class.</param>
        /// <param name="settingsRepository">Settings repository to be used by this class.</param>
        /// <param name="autoAccess">A Boolean value that indicates whether to automatically attempt to gain access if required. Defaults to true.</param>
        public PcehrAccessFilter(IPatientRepository patientRepository, IPcehrViewRepository pcehrViewRepository, IPcehrServiceV2Repository pcehrServiceV2Repository, ISettingsRepository settingsRepository, bool autoAccess = true)
        {
            this.PatientRepository = patientRepository;
            this.PcehrViewRepository = pcehrViewRepository;
            this.PcehrServiceV2Repository = pcehrServiceV2Repository;
            using (settingsRepository)
            {
                this.Settings = settingsRepository.GetSettings();
            }
            this.AutoAccess = autoAccess;
        }

        #endregion Constructors

        #region Methods

        #region Overrides

        /// <summary>
        /// Overrides the OnActionExecuting method to implement the action filter logic.
        /// </summary>
        /// <param name="filterContext"></param>
        public void OnActionExecuting(ActionExecutingContext filterContext)
        {
            //Retrieve required data from context.
            PcehrViewModelBase request = null;
            string hospitalId = null;
            string patientId = null;
            string requestedResource = "";

            //Support different types of parameters:
            //PD View action uses model binding and a single "request" ViewModel parameter.
            //Other actions use individual parameters.

            if (filterContext.ActionParameters.ContainsKey("request"))
            {
                request = (PcehrViewModelBase)filterContext.ActionParameters["request"];
                hospitalId = request.HospitalId;
                patientId = request.PatientId;
            }
            if (filterContext.ActionParameters.ContainsKey("hospitalId") && filterContext.ActionParameters.ContainsKey("patientId"))
            {
                hospitalId = (string)filterContext.ActionParameters["hospitalId"];
                patientId = (string)filterContext.ActionParameters["patientId"];
            }
            if (string.IsNullOrEmpty(hospitalId))
            {
                throw new ArgumentNullException("Expected parameter 'hospitalId' not found.", "hospitalId");
            }
            if (string.IsNullOrEmpty(patientId))
            {
                throw new ArgumentNullException("Expected parameter 'patientId' not found.", "patientId");
            }

            if ((filterContext.HttpContext != null) && (filterContext.HttpContext.Request != null))
            {
                requestedResource = filterContext.HttpContext.Request.Url.AbsolutePath;
            }
            var patientIdentifier = new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem);

            //Check access:
            PcehrAccessManager accessMgr = new PcehrAccessManager(this.PatientRepository, this.PcehrViewRepository, this.PcehrServiceV2Repository);

            var accessResult = accessMgr.AllowAccess(patientIdentifier, this.AutoAccess, ((Controller)filterContext.Controller).GetCurrentUserDetails());

            if (!accessResult.PatientValid && !accessResult.AccessAllowed)
            {
                filterContext.Controller.TempData["PcehrAccessMessage"] = accessResult.AccessMessage;
            }
            if (!accessResult.AccessAllowed && accessResult.PatientValid)
            {
                filterContext.Controller.TempData["PcehrAccessMessage"] = accessResult.AccessMessage;
                string action = requestedResource.StartsWith("/Embedded") ? "EmbeddedGainAccess" : "GainAccess";

                //Redirect to "Gain Access" action.
                filterContext.Result = new RedirectToRouteResult
                    (
                        new RouteValueDictionary()
                        {
                            { "action", action },
                            { "controller", "PcehrView" },
                            { "hospitalId", hospitalId },
                            { "patientId", patientId },
                            { "returnUrl", requestedResource }
                        }
                    );
            }
            // Otherwise, proceed to originally requested function.
        }

        /// <summary>
        /// Overrides the OnActionExecuted method to implement the action filter logic.
        /// </summary>
        /// <param name="filterContext"></param>
        public void OnActionExecuted(ActionExecutedContext filterContext)
        {
            //Do nothing.
        }

        #endregion Overrides

        #endregion Methods
    }
}