﻿using System;
using System.DirectoryServices.AccountManagement;
using System.Security.Principal;
using System.Web.Mvc;
using HIPS.CommonSchemas;
using System.Security.Claims;

namespace HIPS.Web.UI.Helpers
{
    /// <summary>
    /// Common extension methods for MVC Controllers
    /// </summary>
    public static class ControllerExtensions
    {
        public static ActionResult FileInlineInferMime(this Controller c, byte[] data, string filename)
        {
            // Return file based on inferred MIME type from filename (rather than for example AcceptTypes)
            // NB: GetMimeMapping is a .NET 4.5 method
            return c.FileInline(data, System.Web.MimeMapping.GetMimeMapping(filename), filename);
        }

        public static ActionResult FileInline(this Controller c, byte[] data, string contentType, string filename)
        {
            // Attempt to display "inline" (i.e. render in browser rather than file download)
            // See: http://stackoverflow.com/questions/5826649/returning-a-file-to-view-download-in-mvc
            var cd = new System.Net.Mime.ContentDisposition
            {
                FileName = filename,
                Inline = true,
            };
            c.Response.AppendHeader("Content-Disposition", cd.ToString());

            return new FileContentResult(data, contentType);
        }

        public static void SetAjaxErrorResponseCode(this Controller c)
        {
            if (c.Response != null)
            {
                // Unprocessable Entity
                c.Response.StatusCode = 422; // Also prevents OutputCache
                c.Response.TrySkipIisCustomErrors = true;
            }
        }

        public static ActionResult SkipCacheAsError(this Controller c, ActionResult a)
        {
            c.SetAjaxErrorResponseCode();
            return a;
        }

        internal static UserDetails GetCurrentUserDetails(this Controller c)
        {
            ClaimsIdentity cid = null;

            // Attempt to get identity from controller.
            if ((c.User != null) && (c.User.Identity != null))
            {
                // Claims identity created at login
                cid = (ClaimsIdentity)c.User.Identity;
            }

            //If controller identity not present, fall-back to thread identity.
            if (cid == null)
            {
                cid = (ClaimsIdentity)System.Threading.Thread.CurrentPrincipal.Identity;
            }

            if (!cid.IsAuthenticated)
            {
                //throw new InvalidOperationException("User login is required."); // TODO: Handle this differently?
                return new UserDetails()
                {
                    Domain = String.Empty,
                    Login = String.Empty,
                    Name = String.Empty,
                    Role = UserRole.InteractiveUser
                };
            }

            string domain = cid.FindFirst("userDomain").Value; 
            string loginName = cid.Name.Replace(domain + "\\", string.Empty);
            // see if Given Name and Surname were added into the claim on login
            string displayName = loginName.Replace('.', ' ');
            if (cid.FindFirst(ClaimTypes.GivenName) != null && cid.FindFirst(ClaimTypes.Surname) != null)
            {
                if (cid.FindFirst(ClaimTypes.GivenName).Value == cid.FindFirst(ClaimTypes.Surname).Value)
                    displayName = cid.FindFirst(ClaimTypes.GivenName).Value;
                else
                    displayName = cid.FindFirst(ClaimTypes.GivenName).Value + " " + cid.FindFirst(ClaimTypes.Surname).Value;
            }
            
            return new UserDetails()
            {
                Domain = domain,
                Login = loginName,
                Name = displayName,
                Role = UserRole.InteractiveUser
            };
        }

        internal static string GetCurrentUserProfileImageUri(this Controller c)
        {
            ClaimsIdentity cid = null;

            // Attempt to get identity from controller.
            if ((c.User != null) && (c.User.Identity != null))
            {
                // Claims identity created at login
                cid = (ClaimsIdentity)c.User.Identity;
            }

            //If controller identity not present, fall-back to thread identity.
            if (cid == null)
            {
                cid = (ClaimsIdentity)System.Threading.Thread.CurrentPrincipal.Identity;
            }

            // see if Given Name and Surname were added into the claim on login
            string uri = string.Format("{0}{1}", System.Configuration.ConfigurationManager.AppSettings["Account.ProfileImageFilePath"].ToString(), System.Configuration.ConfigurationManager.AppSettings["Account.ProfileImageUriDefault"].ToString()); ;
            if (cid.FindFirst("profileImageUri") != null)
                uri = cid.FindFirst("profileImageUri").Value;

            return uri;
        }
    }
}