﻿#region Using Statements

using System.Linq;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.ServiceContracts.Common;
using HIPS.ServiceContracts.Pcehr.Message;

#endregion Using Statements

namespace HIPS.Web.UI.Helpers
{
    /// <summary>
    /// Represents the result of a check for PCEHR access.
    /// </summary>
    public class PcehrAccessResult
    {
        #region Properties

        /// <summary>
        /// Gets or sets a value that indicates if access to the PCEHR is allowed.
        /// </summary>
        public bool AccessAllowed { get; private set; }

        /// <summary>
        /// Gets or sets a message providing further information on PCEHR access.
        /// </summary>
        public string AccessMessage { get; private set; }

        /// <summary>
        /// Gets a value indicating whether if the patient is valid.
        /// </summary>
        public bool PatientValid { get; private set; }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="accessAllowed">A value that indicates if access to the PCEHR is allowed</param>
        /// <param name="accessMessage">A message providing further information on PCEHR access.</param>
        /// <param name="patientValid">A value that indicates if the patient is valid</param>
        public PcehrAccessResult(bool accessAllowed, string accessMessage, bool patientValid)
        {
            this.AccessAllowed = accessAllowed;
            this.AccessMessage = accessMessage;
            this.PatientValid = patientValid;
        }

        #endregion Constructors
    }

    /// <summary>
    /// Supports checking access to the PCEHR and gaining access automatically if required and possible.
    /// </summary>
    public class PcehrAccessManager
    {
        #region Properties

        /// <summary>
        /// Gets or sets the patient repository to be used by this class.
        /// </summary>
        private HIPS.Web.ModelInterface.Common.IPatientRepository PatientRepository { get; set; }

        /// <summary>
        /// Gets or sets the PCEHR view repository to be used by this class.
        /// </summary>
        private HIPS.Web.ModelInterface.PcehrView.IPcehrViewRepository PcehrViewRepository { get; set; }

        private HIPS.Web.ModelInterface.PcehrView.IPcehrServiceV2Repository PcehrServiceV2Repository { get; set; }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="patientRepository">Patient repository to be used by this class.</param>
        /// <param name="pcehrViewRepository">PCEHR view repository to be used by this class.</param>
        public PcehrAccessManager(HIPS.Web.ModelInterface.Common.IPatientRepository patientRepository, HIPS.Web.ModelInterface.PcehrView.IPcehrViewRepository pcehrViewRepository, HIPS.Web.ModelInterface.PcehrView.IPcehrServiceV2Repository pcehrServiceV2Repository)
        {
            this.PatientRepository = patientRepository;
            this.PcehrViewRepository = pcehrViewRepository;
            this.PcehrServiceV2Repository = pcehrServiceV2Repository;
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Checks the PCEHR to determine if the specified hospital is allowed to access the specified patient's PCEHR.
        /// </summary>
        /// <param name="patientIdentifier">Identifier of the patient (MRN).</param>
        /// <param name="autoAccess">Boolean value indicating whether to automatically attempt to gain access if required.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PcehrAccessResult indicating if the specified hospital is allowed to access the specified patient's PCEHR.</returns>
        public PcehrAccessResult AllowAccess(Mrn patientIdentifier, bool autoAccess, UserDetails operatingUser)
        {
            bool allowAccess = false;
            string message = string.Empty;
            bool patientValid = true;

            // Need to convert patientIdentifier and UserDetails for V2 services
            var patientIdentifierV2 = new HIPS.ServiceContracts.Common.DTO.PatientIdentifier.Mrn();
            var operatingUserV2 = new HIPS.ServiceContracts.Common.DTO.UserIdentity.LocalUser();

            patientIdentifierV2.AlternateOrganisationName = patientIdentifier.AlternateOrganisationName;
            patientIdentifierV2.HospitalCode = patientIdentifier.HospitalCode;
            patientIdentifierV2.HospitalCodeSystem = patientIdentifier.HospitalCodeSystem;
            patientIdentifierV2.Value = patientIdentifier.Value;

            operatingUserV2.Domain = operatingUser.Domain;
            operatingUserV2.FamilyName = operatingUser.Name;
            operatingUserV2.GivenNames = operatingUser.Name;
            operatingUserV2.Login = operatingUser.Login;

            //Check access.
            var pcehrAccessStatus = this.PcehrServiceV2Repository.IsPcehrAdvertisedLocal(
                patientIdentifier: patientIdentifierV2,
                operatingUser: operatingUserV2);

            //Act upon access status.

            if (pcehrAccessStatus != null)
            {
                if (pcehrAccessStatus.IsSuccessful)
                {
                    switch (pcehrAccessStatus.Data.Data.AccessCodeRequired)
                    {
                        case AccessCodeRequired.AccessGranted:
                            {
                                //We already have access!
                                allowAccess = true;
                                break;
                            }
                        case AccessCodeRequired.Unknown:
                        case AccessCodeRequired.WithoutCode:
                            {
                                if (autoAccess)
                                {
                                    // Attempt to gain access without code. If we still can't gain access we'll need to redirect.
                                    var pcehrAccessResult = this.PcehrViewRepository.GainAccessWithoutCode(
                                        patientIdentifier: patientIdentifier,
                                        operatingUser: operatingUser);

                                    if (pcehrAccessResult.IsSuccessful)
                                    {
                                        //Get patient. Groan. Because we HAVE to pass the DOB. Groan some more.
                                        var patient = this.PatientRepository.GetPatientDisclosureDetails(operatingUser, patientIdentifier);
                                        if (patient == null)
                                        {
                                            message = string.Format("An attempt to automatically gain access to the patient's PCEHR was denied. {0}", patient.Messages.ToString());
                                            return new PcehrAccessResult(true, message, false);
                                        }

                                        // Call IsPcehrAdvertised to confirm that Access was granted.
                                        var pcehrIsAdvertised = this.PcehrViewRepository.IsPcehrAdvertised(
                                            patientIdentifier: patientIdentifier,
                                            dateOfBirth: patient.Data.AdmittedPatient.DateOfBirth,
                                            operatingUser: operatingUser);

                                        if (pcehrAccessResult.Data.AccessPermission == PcehrSchemas.GainPcehrAccessStatus.Permit
                                                && pcehrIsAdvertised.IsSuccessful)
                                        {
                                            allowAccess = true;
                                        }
                                        else
                                        {
                                            allowAccess = false;
                                            message = string.Format("An attempt to automatically gain access to the patient's PCEHR was denied. Please try an alternative means for gaining access. {0}", pcehrAccessResult.Messages.AsString());
                                        }
                                    }
                                    else
                                    {
                                        allowAccess = false;
                                        message = string.Format("An attempt to automatically gain access to the patient's PCEHR failed. {0}", pcehrAccessResult.Messages.AsString());

                                        // In case of PCEHR_ERROR_5101, the patient does not have an eHealth record, so the gain access controls should not be displayed.
                                        if (pcehrAccessResult.Messages.Any(a => a.Description.Contains("PCEHR_ERROR_5101")))
                                        {
                                            patientValid = false;
                                        }
                                    }
                                }
                                else
                                {
                                    allowAccess = false;
                                }
                                break;
                            }
                        case AccessCodeRequired.WithCode:
                            {
                                // We can't gain access automatically, so we'll need to redirect.
                                allowAccess = false;
                                message = "A code is required to access the patient's PCEHR.";
                                break;
                            }
                    }
                }
                else
                {
                    // Check the status to see if the patient was valid.
                    if (pcehrAccessStatus.Data.Status == ResponseStatus.Warning)
                    {
                        patientValid = false;
                    }

                    allowAccess = false;
                    message = string.Format("Failed to retrieve the patient's PCEHR status. {0}", pcehrAccessStatus.Messages.AsString());
                }
            }
            else
            {
                allowAccess = false;
                message = "Failed to retrieve the patient's PCEHR status.";
            }

            return new PcehrAccessResult(allowAccess, message, patientValid);
        }

        #endregion Methods
    }
}