using System;
using System.Collections.Generic;
using HIPS.Web.BusinessLogic.Identification;
using HIPS.Web.Model.AssistedRegistration;

namespace HIPS.Web.BusinessLogic.AssistedRegistration
{
    public static class DemographicValidations
    {
        public static List<string> ValidateDemographic(PersonDemographic demographic, bool allowDvaIdentifier)
        {
            List<string> errors = new List<string>();

            // Family Name
            if (!String.IsNullOrWhiteSpace(demographic.FamilyName))
            {
                if (demographic.FamilyName.Length > 100)
                {
                    errors.Add("Family Name cannot exceed 100 characters in length.");
                }
            }
            else
            {
                errors.Add("Family Name is required.");
            }

            // Given Name
            if (!String.IsNullOrWhiteSpace(demographic.GivenName))
            {
                if (demographic.GivenName.Length > 100)
                {
                    errors.Add("Given Name cannot exceed 100 characters in length.");
                }
            }
            else
            {
                errors.Add("Given Name is required.");
            }

            // Date of Birth
            if (demographic.DateOfBirth.HasValue)
            {
                if (demographic.DateOfBirth.Value < DateTime.Now.AddYears(-140))
                {
                    errors.Add("Date of Birth cannot be more than 140 years ago.");
                }

                if (demographic.DateOfBirth.Value > DateTime.Now)
                {
                    errors.Add("Date of Birth cannot be in the future.");
                }
            }
            else
            {
                errors.Add("Date of Birth is required.");
            }

            // Sex
            if (demographic.Sex == null)
            {
                errors.Add("Sex identifier is required.");
            }

            // DVA Identifier
            bool usingDvaIdentifier = !String.IsNullOrWhiteSpace(demographic.DvaFileNumber);
            if (usingDvaIdentifier)
            {
                if (!DvaValidator.IsValidDvaFileNumber(demographic.DvaFileNumber))
                {
                    errors.Add("DVA File Number is invalid.");
                }

                if (!allowDvaIdentifier)
                {
                    errors.Add("DVA File Number identifier functionality is disabled.");
                }
            }

            // Medicare Identifier
            bool usingMedicareIdentifier = !String.IsNullOrWhiteSpace(demographic.MedicareNumber + demographic.MedicareIrn);

            if (usingMedicareIdentifier)
            {
                if (!String.IsNullOrWhiteSpace(demographic.MedicareNumber))
                {
                    errors.AddRange(MedicareValidator.ValidateMedicareNumber(demographic.MedicareNumber, demographic.MedicareIrn));
                }
                else
                {
                    errors.Add("A Medicare IRN cannot be provided without a Medicare Number.");
                }
            }

            // If allowing DVA Identifier
            if (allowDvaIdentifier)
            {
                // Ensure at least DVA or Medicare
                if (!usingDvaIdentifier && !usingMedicareIdentifier)
                {
                    errors.Add("A Medicare Number (and IRN) or a DVA File Number is required.");
                }

                // Ensure not both DVA and Medicare
                //TODO: Reconsider this
                if (usingDvaIdentifier && usingMedicareIdentifier)
                {
                    errors.Add("Cannot provide both a Medicare Number and a DVA File Number.");
                }
            }
            else // If not allowing DVA Identifier
            {
                // Ensure Medicare
                if (!usingMedicareIdentifier)
                {
                    errors.Add("A Medicare Number (and IRN) is required.");
                }
            }

            return errors;
        }
    }
}