﻿using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;

using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.ConsentManagement;
using HIPS.Web.ModelInterface.DocumentManagement;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.ConsentManagement;
using HIPS.Web.UI.ViewModels.Shared;
using System;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for the Data Integrity feature.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class ConsentManagementController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the patient repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientRepository;

        /// <summary>
        /// Gets the consent management service to be used by this controller.
        /// </summary>
        private readonly IConsentManagementService consentManagementService;

        /// <summary>
        /// Gets the episode repository to be used by this controller.
        /// </summary>
        private readonly IEpisodeRepository episodeRepository;

        /// <summary>
        /// Uploaded document repository to be used by this controller.
        /// </summary>
        private readonly IUploadedDocumentRepository uploadedDocumentRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="ConsentManagementController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="consentManagementService">Consent management service to be used by this controller.</param>
        /// <param name="episodeRepository">Episode repository to be used by this controller.</param>
        /// <param name="uploadedDocumentRepository">Uploaded document repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public ConsentManagementController(
            IHospitalRepository hospitalRepository,
            IPatientRepository patientRepository,
            ISettingsRepository settingsRepository,
            IConsentManagementService consentManagementService,
            IEpisodeRepository episodeRepository,
            IUploadedDocumentRepository uploadedDocumentRepository,
            ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.patientRepository = patientRepository;
            this.consentManagementService = consentManagementService;
            this.episodeRepository = episodeRepository;
            this.uploadedDocumentRepository = uploadedDocumentRepository;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets the number of days after discharge to still show an episode on
        /// the list of episodes for which the patient can withdraw consent to
        /// upload documents to their PCEHR.
        /// </summary>
        private int WithdrawConsentEpisodeListDaysDischarged
        {
            get
            {
                string setting = this.Settings.GetSettingValue(Setting.SettingCodes.WithdrawConsentEpisodeListDaysDischarged);
                return int.Parse(setting);
            }
        }

        #endregion Properties

        #region Methods

        #region Actions

        /// <summary>
        /// Display a list of patients for a selected hospital on the Withdraw Consent screen.
        /// </summary>
        /// <param name="lookupMessages">Additional messages to be displayed coming from the lookup MRN action.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Patients(ViewMessageList lookupMessages = null)
        {
            // Create ViewModel:
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            var m = new PatientsViewModel() { HospitalId = hospitalId };
            this.LoadCurrentContext(m);

            // Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                // Update ViewModel with reference data:
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                m.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            // Don't allow load for all patients:
            if (hospitalId != null)
            {
                // Load patients for selected hospital.
                var response = this.patientRepository.ListAdmittedPatients(
                    this.GetCurrentUserDetails(), this.DefaultHospitalCodeSystem, hospitalId, withIhi: null, withPcehr: null, excludeMedicareExclusions: false);

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    if (response.Data.AdmittedPatientList.Count() > 0)
                    {
                        // Update ViewModel with patients.
                        m.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(response.Data.AdmittedPatientList));
                    }
                    else
                    {
                        m.Messages.Add("There are no current patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";
                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    m.Messages.Add(errorMessage, MessageLevel.Error);
                }
                if (lookupMessages != null)
                {
                    m.Messages.AddRange(lookupMessages);
                }
            }
            return this.View("Patients", m);
        }

        /// <summary>
        /// Display a list of episodes for a selected patient on the Withdraw Consent screen.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Episodes(string hospitalId, string patientId)
        {
            var m = new PatientEpisodesViewModel { HospitalId = hospitalId, PatientId = patientId };
            this.LoadCurrentContext(m);
            Mrn patientIdentifier = new Mrn(m.PatientId, m.HospitalId, m.CurrentHospital.CodeSystemCode);
            var response = this.episodeRepository.ListPatientEpisodesInHospital(patientIdentifier, this.GetCurrentUserDetails());
            if (response.IsSuccessful)
            {
                var source = response.Data.PatientEpisodes;
                if (source.Count() > 0)
                {
                    m.PatientEpisodes.AddRange(ObjectMapper.Map<IEnumerable<EpisodeViewModel>>(source));
                }
                else
                {
                    m.Messages.Add(
                        string.Format(
                            "The selected patient has no episodes within the past {0} days.",
                            this.WithdrawConsentEpisodeListDaysDischarged),
                        MessageLevel.Information);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve episodes for the selected patient.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
                m.Messages.Add(errorMessage, MessageLevel.Error);
            }
            return this.View("Episodes", m);
        }

        /// <summary>
        /// Look up a patient by MRN for withdrawal of consent. If the patient
        /// is found and has more than one recent episode, display the list of
        /// episodes. If the patient is found and has only one recent episode
        /// then show the appropriate form to withdraw or reinstate consent for
        /// that episode. If the patient is not found, redisplay the list of
        /// patients with a message indicating that the patient was not found.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="lookupMrn">User-entered identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult LookupMrn(string hospitalId, string lookupMrn)
        {
            var m = new PatientEpisodesViewModel { HospitalId = hospitalId, PatientId = lookupMrn };
            this.LoadCurrentContext(m);
            if (m.CurrentPatient != null && m.CurrentPatient.AdmissionDate != DateTime.MinValue)
            {
                if (m.CurrentPatient.EpisodeCount > 1)
                {
                    return this.Episodes(hospitalId, lookupMrn);
                }
                else if (m.CurrentPatient.ConsentWithdrawn == true)
                {
                    return this.ReinstateEpisode(hospitalId, lookupMrn, m.CurrentPatient.EpisodeId);
                }
                else
                {
                    return this.WithdrawEpisode(hospitalId, lookupMrn, m.CurrentPatient.EpisodeId);
                }
            }
            else
            {
                ViewMessageList messages = new ViewMessageList();

                if (m.CurrentPatient != null)
                {
                    messages.Add(string.Format("There are no episodes for patient {0}.", m.CurrentPatient.MRN), MessageLevel.Error);
                }
                else
                {
                    messages.Add(string.Format("There is no current patient with MRN {0}.", m.PatientId), MessageLevel.Error);
                }
                
                return this.Patients(messages);
            }
        }

        /// <summary>
        /// Display a form to withdraw consent to upload documents for a selected episode.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="episodeId">Identifier of the episode.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult WithdrawEpisode(string hospitalId, string patientId, string episodeId)
        {
            var model = new WithdrawConsentViewModel(hospitalId, patientId, episodeId);
            this.LoadCurrentContext(model);
            this.LoadUploadedDocumentCount(model);
            return this.View("WithdrawEpisode", model);
        }

        /// <summary>
        /// Display a form to reinstate consent to upload documents for a selected episode.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="episodeId">Identifier of the episode.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult ReinstateEpisode(string hospitalId, string patientId, string episodeId)
        {
            var m = new WithdrawConsentViewModel(hospitalId, patientId, episodeId);
            this.LoadCurrentContext(m);
            return this.View("ReinstateEpisode", m);
        }

        /// <summary>
        /// Process the AJAX request to withdraw consent to upload documents for a selected episode.
        /// </summary>
        /// <param name="model">Contents of the form submission.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        public ActionResult WithdrawEpisode(WithdrawConsentViewModel model)
        {
            return this.WithdrawOrReinstate(model, true);
        }

        /// <summary>
        /// Process the AJAX request to reinstate consent to upload documents for a selected episode.
        /// </summary>
        /// <param name="model">Contents of the form submission.</param>
        /// <returns>The JSON result.</returns>
        [HttpPost]
        public ActionResult ReinstateEpisode(WithdrawConsentViewModel model)
        {
            return this.WithdrawOrReinstate(model, false);
        }

        /// <summary>
        /// Process the AJAX request to withdraw or reinstate consent to upload documents for a selected episode.
        /// </summary>
        /// <param name="model">Contents of the form submission.</param>
        /// <param name="consentWithdrawn">Whether consent is being withdrawn (true) or reinstated (false).</param>
        /// <returns>The JSON result.</returns>
        private ActionResult WithdrawOrReinstate(WithdrawConsentViewModel model, bool consentWithdrawn)
        {
            // As returning JSON ensure the request was AJAX.
            if (!Request.IsAjaxRequest())
            {
                return this.Content("JavaScript is required for this functionality.");
            }

            // Check binding validation errors
            if (!ModelState.IsValid)
            {
                return this.Json(new { Echo = model, Errors = ModelState.ToErrorDictionary() });
            }

            this.LoadCurrentContext(model);

            Mrn patientIdentifier = new Mrn(model.PatientId, model.HospitalId, model.CurrentHospital.CodeSystemCode);

            ServiceResponse<bool> response = this.consentManagementService.RecordConsent(consentWithdrawn, model.Notes, patientIdentifier, model.CurrentEpisode.AdmissionDate, this.GetCurrentUserDetails());

            return this.Json(new { Response = response, Echo = model, Errors = ModelState.ToErrorDictionary() });
        }

        #endregion Actions

        #region Helpers

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        private void LoadCurrentContext(ConsentManagementViewModelBase model)
        {
            var mrn = new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem);
            var user = this.GetCurrentUserDetails();

            // Load current hospital.
            if (!string.IsNullOrEmpty(model.HospitalId))
            {
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);
                }
            }

            // Load current patient.
            if (!string.IsNullOrEmpty(model.PatientId))
            {
                var response = this.patientRepository.GetPatientDisclosureDetails(this.GetCurrentUserDetails(), mrn);
                model.CurrentPatient = ObjectMapper.Map<PatientViewModel>(response.Data.AdmittedPatient);
            }

            // Load current episode.
            if (!string.IsNullOrEmpty(model.EpisodeId))
            {
                model.CurrentEpisode = ObjectMapper.Map<EpisodeViewModel>(this.episodeRepository.GetEpisodeDetails(user, model.EpisodeId, mrn).Data);
            }
        }

        /// <summary>
        /// Loads the count of uploaded documents and writes messages into the
        /// model to explain why consent cannot be withdrawn until all uploaded
        /// documents for this episode have been removed.
        /// </summary>
        /// <param name="model">Withdraw consent view model.</param>
        private void LoadUploadedDocumentCount(WithdrawConsentViewModel model)
        {
            var mrn = new Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem);
            var result = this.uploadedDocumentRepository.List(this.GetCurrentUserDetails(), null, mrn);
            var episodeResponse = this.episodeRepository.ListPatientEpisodesInHospital(mrn, this.GetCurrentUserDetails());

            if (result.IsSuccessful && episodeResponse.IsSuccessful)
            {
                model.UploadedDocumentCount = result.Data.DocumentList
                    .Count(d => d.Status == PcehrSchemas.LocalDocumentStatus.Uploaded && episodeResponse.Data.PatientEpisodes.Any(i => i.SourceSystemEpisodeId == model.EpisodeId && i.AdmissionDate == d.AdmissionDate));

                var firstUpload = result.Data.DocumentList
                    .OrderBy(d => d.UploadedDate)
                    .FirstOrDefault(d => d.Status == PcehrSchemas.LocalDocumentStatus.Uploaded);

                // Add Error Message when there are multiple Episodes that has the same Admission Date.
                if (firstUpload != null)
                {
                    var episodeList = episodeResponse.Data.PatientEpisodes.Where(i => i.AdmissionDate == firstUpload.AdmissionDate);
                    if (episodeList.Count() > 1 && episodeList.Any(i => i.SourceSystemEpisodeId == model.EpisodeId))
                    {
                        model.Messages.Add("Consent cannot be withdrawn for this episode because this episode has the same admission date/time as another episodes.", MessageLevel.Error);
                    }
                }

                if (model.UploadedDocumentCount >= 1)
                {
                    model.Messages.Add(string.Format("Consent cannot be withdrawn for this episode because the {0} was uploaded {1:dd/MM/yyyy HH:mm}.", firstUpload.DocumentType, firstUpload.UploadedDate), MessageLevel.Error);
                    model.Messages.Add("If the patient asked to withdraw consent after the document was uploaded, the authoring organisation is not obliged to remove the document. The patient may choose to remove the document using the consumer portal or via Medicare channels.", MessageLevel.Information);
                    model.Messages.Add("The authoring organisation is obliged to remove the document if it  did not have consent at the time it uploaded the original document. This could happen if the document was uploaded after the patient asked to withdraw consent, but the patient's request had not yet been administered in the clinical system.", MessageLevel.Information);
                }

            }
        }

        #endregion Helpers

        #endregion Methods
    }
}