﻿using System.Linq;

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;

using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.PcehrView;
using System;

namespace HIPS.Web.UI.Helpers
{
    /// <summary>
    /// Extension methods for working with repositories.
    /// </summary>
    public static class RepositoryExtensions
    {
        #region Methods

        /// <summary>
        /// Gets a specified patient in hospital.
        /// </summary>
        /// <param name="r">IPatientRepository.</param>
        /// <param name="patientIdentifier">Identifier of the patient (MRN).</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientInHospital</returns>
        internal static PatientSchemas.PatientInHospital GetPatientInHospital(this IPatientRepository r, Mrn patientIdentifier, UserDetails operatingUser)
        {
            // Load patients for specified hospital.
            var patients = r.ListPatientsCurrentlyInHospital(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, true, operatingUser);

            // Ensure loading was successful.
            if (patients.IsSuccessful)
            {
                return patients.Data.PatientInHospitalList.SingleOrDefault(p => p.Mrn == patientIdentifier.Value);
            }

            throw new System.Exception("Could not retrieve specified patient.");
        }

        /// <summary>
        /// Gets a specified patient in hospital.
        /// </summary>
        /// <param name="r">IPatientRepository.</param>
        /// <param name="patientIdentifier">Identifier of the patient (MRN).</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientInHospital</returns>
        internal static PatientSchemas.PatientInHospital GetPatientInHospital(this IPatientRepository r, Mrn patientIdentifier, bool? withPcehr, UserDetails operatingUser)
        {
            // Load patients for specified hospital.
            var patients = r.ListPatientsCurrentlyInHospital(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, withPcehr, operatingUser);

            // Ensure loading was successful.
            if (patients.IsSuccessful)
            {
                return patients.Data.PatientInHospitalList.SingleOrDefault(p => p.Mrn == patientIdentifier.Value);
            }

            throw new System.Exception("Could not retrieve specified patient.");
        }

        /// <summary>
        /// Gets a specified patient in hospital.
        /// </summary>
        /// <param name="r">IPatientRepository.</param>
        /// <param name="patientIdentifier">Identifier of the patient (MRN).</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientInHospital</returns>
        internal static PatientSchemas.PatientInHospital GetPatientInHospital(this IPatientRepository r, RegisteredEnterprisePatient patientIdentifier, bool? withPcehr, UserDetails operatingUser)
        {
            // Load patients for specified hospital.
            var patients = r.ListPatientsCurrentlyInHospital(patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, withPcehr, operatingUser);

            // Ensure loading was successful.
            if (patients.IsSuccessful)
            {
                return patients.Data.PatientInHospitalList.SingleOrDefault(p => p.Mrn == patientIdentifier.Mrn);
            }

            throw new System.Exception("Could not retrieve specified patient.");
        }

        internal static PatientSchemas.AdmittedPatient GetAdmittedPatient(this IPatientRepository r, Mrn patientIdentifier, UserDetails operatingUser)
        {
            // Load patients for specified hospital.
            var patients = r.ListAdmittedPatients(operatingUser, patientIdentifier.HospitalCodeSystem, patientIdentifier.HospitalCode, null, null, false);

            // Ensure loading was successful.
            if (patients.IsSuccessful)
            {
                return patients.Data.AdmittedPatientList.SingleOrDefault(p => p.Mrn == patientIdentifier.Value);
            }

            throw new System.Exception("Could not retrieve specified patient.");
        }

        /// <summary>
        /// Gets metadata for a specified document.
        /// </summary>
        /// <param name="r">IPcehrViewRepository.</param>
        /// <param name="patientIdentifier">Identifier of the patient (MRN).</param>
        /// <param name="repositoryId">Identifier of the repository.</param>
        /// <param name="documentId">Identifier of the document.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DocumentMetaDataItem</returns>
        internal static DocumentMetaDataItem GetDocumentMetaData(this IPcehrViewRepository r, Mrn patientIdentifier, string repositoryId, string documentId, UserDetails operatingUser, DateTime? fromDate, DateTime? toDate)
        {
            // Load documents for specified patient.
            var docs = r.ListActiveDocuments(patientIdentifier, operatingUser, fromDate, toDate, null, null);
           
            // Ensure loading was successful.
            if (docs.IsSuccessful)
            {
                if (docs.Data.DocumentList.Count > 0)
                {
                    return docs.Data.DocumentList.Single(d => (d.RepositoryUniqueId == repositoryId) && (d.DocumentUniqueId == documentId));
                }
            }

            throw new System.Exception("Could not retrieve specified document.");
        }

        #endregion Methods
    }
}