﻿using System.ComponentModel.DataAnnotations;

using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.DocumentUpload
{
    /// <summary>
    /// Base view model to be used with the discharge summary upload feature.
    /// </summary>
    public class DocumentUploadViewModelBase
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="DocumentUploadViewModelBase" /> class.
        /// </summary>
        public DocumentUploadViewModelBase()
        {
            this.Messages = new ViewMessageList();
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the list of messages associated with the view model.
        /// </summary>
        public ViewMessageList Messages { get; set; }

        /// <summary>
        /// Gets or sets the identifier (code) associated with the selected hospital.
        /// </summary>
        [Required]
        [Display(Name = "Hospital")]
        public string HospitalId { get; set; }

        /// <summary>
        /// Gets or sets the identifier (StatePatientId) associated with the selected patient.
        /// </summary>
        public string PatientId { get; set; }

        /// <summary>
        /// Gets or sets the identifier (MRN) associated with the selected patient.
        /// </summary>
        public string StatePatientId { get; set; }

        /// <summary>
        /// Gets or sets the identifier (visit number) associated with the selected episode.
        /// </summary>
        public string EpisodeId { get; set; }

        /// <summary>
        /// Gets or sets the Set ID of the document, which is common to all versions of the document.
        /// </summary>
        public string SetId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected hospital.
        /// </summary>
        public HospitalViewModel CurrentHospital { get; set; }

        /// <summary>
        /// Gets or sets the currently selected patient.
        /// </summary>
        public PatientViewModel CurrentPatient { get; set; }

        /// <summary>
        /// Gets or sets the currently selected episode.
        /// </summary>
        public EpisodeViewModel CurrentEpisode { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Determines if the view model contains a selected hospital.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected hospital, false otherwise.</returns>
        public bool IsHospitalSelected()
        {
            return !string.IsNullOrEmpty(this.HospitalId);
        }

        /// <summary>
        /// Determines if the view model contains a selected patient.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected patient, false otherwise.</returns>
        public bool IsPatientSelected()
        {
            return this.IsHospitalSelected() && (!string.IsNullOrEmpty(this.PatientId)) && this.CurrentPatient != null;
        }

        #endregion Methods
    }
}