using System;

namespace HIPS.Web.Components.Cache
{
    /// <summary>
    /// A cache provider object that does not implement caching.
    /// Useful if disabling caching is desired.
    /// </summary>
    public class NoCachingCacheProvider : CacheProvider
    {
        /// <summary>
        /// Used for double-checked locking.
        /// </summary>
        private readonly object cacheLock = new object();

        /// <summary>
        /// Initialises a new instance of the <see cref="NoCachingCacheProvider"/> class.
        /// </summary>
        public NoCachingCacheProvider()
            : base()
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="NoCachingCacheProvider"/> class.
        /// </summary>
        /// <param name="defaultAbsoluteExpirationOffset">The default absolute expiration offset, or null if not using absolute expiration.</param>
        /// <param name="defaultSlidingExpiration">The default sliding expiration, or null if not using default sliding expiration.</param>
        public NoCachingCacheProvider(TimeSpan? defaultAbsoluteExpirationOffset, TimeSpan? defaultSlidingExpiration = null)
            : base(defaultAbsoluteExpirationOffset, defaultSlidingExpiration)
        {
        }

        /// <summary>
        /// Returns null.
        /// </summary>
        /// <typeparam name="T">Type of the item to be retrieved from the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <returns>
        /// The cached value for the specified key, or null if no value found in the cache.
        /// </returns>
        public override T Get<T>(string cacheKey)
        {
            return null;
        }

        /// <summary>
        /// Does nothing.
        /// </summary>
        /// <typeparam name="T">Type of the item to be stored in the cache.</typeparam>
        /// <param name="cacheKey">A unique identifier for the cache entry.</param>
        /// <param name="item">The item to be stored in the cache.</param>
        /// <param name="slidingExpiration">A value indicating when a cache entry should be evicted if it has not been accessed in a given span of time.</param>
        /// <param name="absoluteExpiration">A value indicating whether a cache entry should be evicted at a specified date and time.</param>
        /// <param name="dependencyKeys">If supported, an optional set of keys that this entry is dependent on. If these keys do not exist in the cache, or are changed or removed, this entry is removed from the cache.</param>
        public override void Set<T>(string cacheKey, T item, TimeSpan slidingExpiration, DateTimeOffset absoluteExpiration, params string[] dependencyKeys)
        {
            // Do Nothing
        }

        /// <summary>
        /// Returns a reference to an object that can be used as an applicable thread lock for this cache for the specified key.
        /// This is useful for functionality such as double-check locking.
        ///  </summary>
        /// <param name="cacheKey">The cache key.</param>
        /// <returns>A cache-specific key-specific thread lock.</returns>
        public override object GetLock(string cacheKey)
        {
            return this.cacheLock;
        }
    }
}