﻿using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.IO;
using System.Linq;
using System.Web;
using System.Web.Mvc;

namespace HIPS.Web.Components.Validation
{
    [AttributeUsage(AttributeTargets.Property, AllowMultiple = false)]
    public sealed class FileExtensionAttribute : ValidationAttribute, IClientValidatable
    {
        private string extensions;

        public string Extensions
        {
            get { return string.IsNullOrWhiteSpace(extensions) ? ".png|.jpg|.jpeg|.gif|.tiff|.pdf" : extensions; }
            set { extensions = value; }
        }

        public override bool IsValid(object value)
        {
            if (value != null)
            {
                if (value is HttpPostedFileBase)
                {
                    var file = value as HttpPostedFileBase;
                    return this.ValidateFileExtension(file);
                }
                else if (value is IEnumerable<HttpPostedFileBase>)
                {
                    var files = value as IEnumerable<HttpPostedFileBase>;

                    foreach (var file in files)
                    {
                        if (file != null)
                        {
                            return this.ValidateFileExtension(file);
                        }
                    }
                }
            }

            return true;
        }

        public override string FormatErrorMessage(string name)
        {
            var extensionFormatted = string.Join(", ", this.Extensions.Trim().Replace(".", string.Empty).Split('|')).ToUpperInvariant();

            return string.Format("{0} only accepts {1} file(s).", name, extensionFormatted);
        }

        public IEnumerable<ModelClientValidationRule> GetClientValidationRules(ModelMetadata metadata, ControllerContext context)
        {
            var rule = new ModelClientValidationRule()
            {
                ValidationType = "fileextensions",
                ErrorMessage = FormatErrorMessage(metadata.GetDisplayName())
            };

            rule.ValidationParameters.Add("ext", this.Extensions.Trim().Replace(".", string.Empty));
            yield return rule;
        }

        private bool ValidateFileExtension(HttpPostedFileBase file)
        {
            string fileName = file.FileName;
            string extension = Path.GetExtension(fileName);

            return this.Extensions.Trim().ToUpperInvariant().Split('|').Contains(extension.ToUpperInvariant());
        }
    }
}