﻿using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Web;
using System.Web.Mvc;

namespace HIPS.Web.Components.Validation
{
    [AttributeUsage(AttributeTargets.Property, AllowMultiple = false)]
    public class FileSizeAttribute : ValidationAttribute, IClientValidatable
    {
        private const int SCALE = 1024;

        private long? minimumSize;
        private long? maximumSize;

        public long MinimumSize
        {
            get { return !minimumSize.HasValue ? SCALE : minimumSize.Value; }
            set { minimumSize = value; }
        }

        public long MaximumSize
        {
            get { return !maximumSize.HasValue ? (10 * (long)Math.Pow(SCALE, 2)) : maximumSize.Value; }
            set { maximumSize = value; }
        }

        public override bool IsValid(object value)
        {
            if (value as HttpPostedFileBase != null)
            {
                var file = value as HttpPostedFileBase;
                return this.ValidateFileSize(file);
            }
            else if (value as IEnumerable<HttpPostedFileBase> != null)
            {
                var files = value as IEnumerable<HttpPostedFileBase>;

                foreach (var file in files)
                {
                    if (file != null)
                    {
                        return this.ValidateFileSize(file);
                    }
                }
            }

            return true;
        }

        public override string FormatErrorMessage(string name)
        {
            return string.Format("The file size must be between {0} and {1}.", this.FormatBytes(this.MinimumSize), this.FormatBytes(this.MaximumSize));
        }

        public IEnumerable<ModelClientValidationRule> GetClientValidationRules(ModelMetadata metadata, ControllerContext context)
        {
            var rule = new ModelClientValidationRule()
            {
                ValidationType = "filesize",
                ErrorMessage = FormatErrorMessage(metadata.GetDisplayName())
            };

            rule.ValidationParameters.Add("min", this.MinimumSize);
            rule.ValidationParameters.Add("max", this.MaximumSize);

            yield return rule;
        }

        private bool ValidateFileSize(HttpPostedFileBase file)
        {
            var size = file.ContentLength;

            return (size >= this.MinimumSize && size <= this.MaximumSize);
        }

        private string FormatBytes(long bytes)
        {
            const int scale = 1024;
            string[] orders = new string[] { "GB", "MB", "KB", "Bytes" };
            long max = (long)Math.Pow(scale, orders.Length - 1);

            foreach (string order in orders)
            {
                if (bytes > max)
                    return string.Format("{0:##.##} {1}", decimal.Divide(bytes, max), order);

                max /= scale;
            }
            return "0 Bytes";
        }
    }
}