﻿using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Web.Mvc;

namespace HIPS.Web.Components.Validation
{
    [AttributeUsage(AttributeTargets.Property, AllowMultiple = false)]
    public class LuhnCheckDigitValidator : ValidationAttribute, IClientValidatable
    {
        private string healthIdentifier;

        protected override ValidationResult IsValid(object value, ValidationContext validationContext)
        {
            if (value is string)
            {
                string healthIdentifierWithLuhnCheckDigit = value.ToString().Replace(" ", string.Empty).Replace("-", string.Empty);

                try
                {
                    healthIdentifier = healthIdentifierWithLuhnCheckDigit;
                    if (healthIdentifierWithLuhnCheckDigit.Length > 15)
                    {
                        healthIdentifier = healthIdentifierWithLuhnCheckDigit.Substring(0, 15);
                    }

                    bool result = this.CalculateLuhnCheckDigit(healthIdentifier) == healthIdentifierWithLuhnCheckDigit;
                    if (result)
                    {
                        return ValidationResult.Success;
                    }
                }
                catch (Exception ex)
                {
                    return new ValidationResult(ex.Message);
                }
            }

            if (value == null)
            {
                return ValidationResult.Success;
            }

            return new ValidationResult("Invalid Health Identifier.");
        }

        private string CalculateLuhnCheckDigit(string healthIdentifier)
        {
            string result = string.Empty;
            int num = 0;
            bool flag = true;

            if (healthIdentifier == null)
            {
                throw new ArgumentException("INVALID Health Identifier - null value");
            }

            healthIdentifier = healthIdentifier.Replace(" ", string.Empty).Replace("-", string.Empty);
            if (healthIdentifier.Length == 15)
            {
                try
                {
                    for (int i = 14; i >= 0; i--)
                    {
                        int num2 = Convert.ToInt32(healthIdentifier.Substring(i, 1));
                        if (flag)
                        {
                            num2 *= 2;
                            if (num2 > 9)
                            {
                                num2 -= 9;
                            }
                        }

                        flag = !flag;
                        num += num2;
                    }

                    int num3 = Convert.ToInt32(num.ToString().Substring(num.ToString().Length - 1, 1));
                    if (num3 > 0)
                    {
                        num3 = 10 - num3;
                    }

                    result = healthIdentifier + num3.ToString();
                    return result;
                }
                catch (Exception)
                {
                    throw new Exception(string.Format("ERROR calcualting health identifier LUNH check digit: {0}", healthIdentifier));
                }
            }

            throw new ArgumentException(string.Format("INVALID Health Identifier - must be 15 digits : {0}", healthIdentifier));
        }

        public IEnumerable<ModelClientValidationRule> GetClientValidationRules(ModelMetadata metadata, ControllerContext context)
        {
            var rule = new ModelClientValidationRule();
            rule.ErrorMessage = this.FormatErrorMessage(metadata.GetDisplayName());
            rule.ValidationType = "luhncheck";

            yield return rule;
        }
    }
}