﻿using System.Collections.Generic;
using System.Web.Mvc;
using System.Web.Routing;

namespace HIPS.Web.Components.Web
{
    /// <summary>
    /// Extension methods for working with routes.
    /// </summary>
    public static class RoutingExtensions
    {
        #region Declarations

        /// <summary>
        /// Key used when setting/getting the title associated with a route.
        /// </summary>
        private const string ROUTE_TITLE_KEY = "__RouteTitle";

        #endregion Declarations

        #region Methods

        /// <summary>
        /// Get the title for a route within the RouteCollection located based on the specified parameters.
        /// </summary>
        /// <param name="routes">The RouteCollection to locate the route within.</param>
        /// <param name="actionName">Name of the action to match against the route.</param>
        /// <param name="controllerName">Name of the controller to match against the route. Pass null to derive from the RequestContext.</param>
        /// <param name="routeValues">An object containing the parameters for the route.</param>
        /// <param name="requestContext">The context in which the request is being made.</param>
        /// <param name="model">Object representing the view model. Pass null if not relevant.</param>
        /// <returns>String representing the title for route.</returns>
        public static string GetRouteTitle(this RouteCollection routes, string actionName, string controllerName, object routeValues, RequestContext requestContext, object model)
        {
            //Find the route within the collection based on the specified parameters.
            var route = routes.GetRoute(actionName, controllerName, routeValues, requestContext).Route as Route;
            if (route == null)
            {
                throw new System.Exception("Unable to find a route matching the specified values.");
            }
            //Return the title for the route.
            return route.GetTitle(model);
        }

        /// <summary>
        /// Get the title for a route.
        /// </summary>
        /// <param name="route">The route to get the title for.</param>
        /// <returns>String representing the title for the route.</returns>
        public static string GetTitle(this Route route)
        {
            return route.GetTitle(null);
        }

        /// <summary>
        /// Get the title for a route.
        /// </summary>
        /// <param name="route">Route to get the title for.</param>
        /// <param name="model">Object representing the view model. Pass null if not relevant.</param>
        /// <returns>String representing the title for the route.</returns>
        public static string GetTitle(this Route route, object model)
        {
            string result = string.Empty;

            // Attempt to find the title object in the route's DataTokens collection.
            if (route.DataTokens.ContainsKey(ROUTE_TITLE_KEY))
            {
                //Retrieve the title object.
                object title = route.DataTokens[ROUTE_TITLE_KEY];
                if (title is string)
                {
                    //Simply return the string title value.
                    result = (string)title;
                }
                if (title is System.Func<object, string>)
                {
                    //Invoke the System.Func that generates the title, passing the provided model parameter.
                    result = ((System.Func<object, string>)title)(model);
                }
            }

            return result;
        }

        /// <summary>
        /// Set the title for a route.
        /// </summary>
        /// <param name="route">Route to set the title for.</param>
        /// <param name="title">Object representing the title for the route. May be either a string or a <see cref="System.Func"/> accepting a single object parameter representing the view model and returning a string.</param>
        /// <returns>The given Route object, for fluency.</returns>
        public static Route SetTitle(this Route route, object title)
        {
            //Store the specified title object in the route's DataTokens collection.
            route.DataTokens[ROUTE_TITLE_KEY] = title;
            return route;
        }

        /// <summary>
        /// Gets the virtual path data information for a route using route data information and the current context.
        /// </summary>
        /// <param name="routes">The RouteCollection to locate the route within.</param>
        /// <param name="actionName">Name of the action to match against the route.</param>
        /// <param name="controllerName">Name of the controller to match against the route. Pass null to derive from the RequestContext.</param>
        /// <param name="routeValues">An object containing the parameters for the route.</param>
        /// <param name="requestContext">The context in which the request is being made.</param>
        /// <returns>Virtual path data.</returns>
        /// <remarks>
        /// Adapted from MVC Url Helpers.
        /// </remarks>
        public static VirtualPathData GetRoute(this RouteCollection routes, string actionName, string controllerName, object routeValues, RequestContext requestContext)
        {
            RouteValueDictionary values = RoutingExtensions.MergeRouteValues(actionName, controllerName, requestContext.RouteData.Values, new RouteValueDictionary(routeValues), true);
            return routes.GetVirtualPathForArea(requestContext, null, values);
        }

        /// <summary>
        /// Merges route data using the current context and provided route values.
        /// </summary>
        /// <param name="actionName"></param>
        /// <param name="controllerName"></param>
        /// <param name="implicitRouteValues"></param>
        /// <param name="routeValues"></param>
        /// <param name="includeImplicitMvcValues"></param>
        /// <remarks>
        /// Adapted from MVC RouteValueHelpers.
        /// </remarks>
        internal static RouteValueDictionary MergeRouteValues(string actionName, string controllerName, RouteValueDictionary implicitRouteValues, RouteValueDictionary routeValues, bool includeImplicitMvcValues)
        {
            RouteValueDictionary routeValueDictionary = new RouteValueDictionary();
            if (includeImplicitMvcValues)
            {
                object obj;
                if (implicitRouteValues != null && implicitRouteValues.TryGetValue("action", out obj))
                    routeValueDictionary["action"] = obj;
                if (implicitRouteValues != null && implicitRouteValues.TryGetValue("controller", out obj))
                    routeValueDictionary["controller"] = obj;
            }
            if (routeValues != null)
            {
                foreach (KeyValuePair<string, object> keyValuePair in new RouteValueDictionary(routeValues))
                    routeValueDictionary[keyValuePair.Key] = keyValuePair.Value;
            }
            if (actionName != null)
                routeValueDictionary["action"] = actionName;
            if (controllerName != null)
                routeValueDictionary["controller"] = controllerName;
            return routeValueDictionary;
        }

        #endregion Methods
    }
}