﻿using System;
using HIPS.Client.Proxy;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.ServiceContracts.Pcehr.Message;
using HIPS.ServiceContracts.Common.DTO.PatientIdentifier;
using HIPS.ServiceContracts.Common.DTO.UserIdentity;

namespace HIPS.Web.Data.Hips.PcehrView
{
    /// <summary>
    /// Implements a repository for viewing the contents of a PCEHR.
    /// </summary>
    public class PcehrServiceV2Repository : RepositoryBase<PcehrServiceV2Proxy>, HIPS.Web.ModelInterface.PcehrView.IPcehrServiceV2Repository
    {
        #region Constructors

        /// <summary>
        /// Initializes a new instance of the <see cref="PcehrViewRepository"/> class.
        /// </summary>
        public PcehrServiceV2Repository()
            : this(new HIPS.Web.Components.Cache.NoCachingCacheProvider(), null)
        {
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PcehrViewRepository"/> class.
        /// </summary>
        /// <param name="cacheProvider">Cache provider to be employed by the repository.</param>
        /// <param name="cacheKeyPrefix">Key prefix to be employed for caching.</param>
        public PcehrServiceV2Repository(ICacheProvider cacheProvider, string cacheKeyPrefix = "")
            : base(cacheProvider, cacheKeyPrefix)
        {
            //Initialise client proxy ready for use.
            this.ClientProxy = new PcehrServiceV2Proxy("PCEHREndPointV2");
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Retrieves details of the patient's PCEHR status.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="dateOfBirth">Date of birth for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DoesPcehrExistResponse containing the results of the operation.</returns>
        public ServiceResponse<IsPcehrAdvertisedLocalResponse> IsPcehrAdvertisedLocal(Mrn patientIdentifier, UserBase operatingUser)
        {
            return this._IsPcehrAdvertised(patientIdentifier, operatingUser);
        }

        #region Private

        /// <summary>
        /// Retrieves details of the patient's PCEHR status.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="dateOfBirth">Date of birth for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DoesPcehrExistResponse containing the results of the operation.</returns>
        private ServiceResponse<IsPcehrAdvertisedLocalResponse> _IsPcehrAdvertised(Mrn patientIdentifier, UserBase operatingUser)
        {
            IsPcehrAdvertisedLocalResponse result;
            ResponseMessageList messages = new ResponseMessageList();
            bool isSuccessful = false;

            var request = new IsPcehrAdvertisedLocalRequest();

            request.PatientIdentifier = patientIdentifier;
            request.User = operatingUser;

            try
            {
                result = this.ClientProxy.IsPcehrAdvertisedLocal(request);
            }
            catch(System.ServiceModel.FaultException<HIPS.ServiceContracts.Common.Fault.ServiceOperationFault> ex)
            {
                result = new IsPcehrAdvertisedLocalResponse { Status = ServiceContracts.Common.ResponseStatus.Warning };
                messages.Add(ex.Detail.Message, MessageLevel.Error);
            }

            if (result != null)
            {
                // HIPS Status should be OK
                isSuccessful = result.Status == ServiceContracts.Common.ResponseStatus.OK;
            }
            else
            {
                if (messages.Count < 1)
                {
                    // Add a message if unable to get a response (shouldn't really happen)
                    messages.Add("Unable to retrieve documents for the selected patient.", MessageLevel.Error);
                }
            }

            return new ServiceResponse<IsPcehrAdvertisedLocalResponse>(result, isSuccessful, messages, DateTime.Now);
        }

        #endregion Private

        #endregion Methods
    }
}