using System;
using System.ComponentModel.DataAnnotations;

namespace HIPS.Web.UI.ViewModels.AssistedRegistration
{
    /// <summary>
    /// View model for unregistered patient.
    /// </summary>
    public class UnregisteredPatientViewModel
    {
        #region Properties

        /// <summary>
        /// Gets or sets the Hospital Id.
        /// </summary>
        public string HospitalId { get; set; }

        /// <summary>
        /// Gets or sets the MRN of the patient.
        /// </summary>
        public string Mrn { get; set; }

        /// <summary>
        /// Gets or sets the IHI of the patient.
        /// </summary>
        public string Ihi { get; set; }

        /// <summary>
        /// Gets or sets the Family name of the patient.
        /// </summary>
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets the Given names of the patient.
        /// </summary>
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets the date of birth of the patient.
        /// </summary>
        [DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets the ward of the patient's most recent episode.
        /// </summary>
        public string Ward { get; set; }

        /// <summary>
        /// Gets or sets the room of the patient's most recent episode.
        /// </summary>
        public string Room { get; set; }

        /// <summary>
        /// Gets or sets the bed of the patient's most recent episode.
        /// </summary>
        public string Bed { get; set; }

        /// <summary>
        /// Gets the location (ward:room:bed) of the patient.
        /// </summary>
        public string Location
        {
            get
            {
                string location = string.Join(":", new[] { this.Ward, this.Room, this.Bed });
                return location == "::" ? "Unknown" : location;
            }
        }

        /// <summary>
        /// Gets the IHI of the patient formatted for display.
        /// </summary>
        public string FormattedIhi
        {
            get
            {
                return System.Text.RegularExpressions.Regex.Replace(this.Ihi, @"(\d{4})(\d{4})(\d{4})(\d{4})", @"$1 $2 $3 $4");
            }
        }

        #endregion Properties

        #region View Logic

        /// <summary>
        /// Gets Valid registration methods.
        /// </summary>
        public RegistrationMethods ValidRegistrationMethods
        {
            get
            {
                // Unknown DOB (Unregistrable)
                if (!this.DateOfBirth.HasValue)
                {
                    return RegistrationMethods.None;
                }

                // Under 14
                if (this.DateOfBirth > DateTime.Now.AddYears(-14))
                {
                    return RegistrationMethods.DependantOnly;
                }

                // 14 or over, under 18
                if (this.DateOfBirth > DateTime.Now.AddYears(-18))
                {
                    return RegistrationMethods.StandardOrDependant;
                }

                // 18 or over
                return RegistrationMethods.StandardOnly;
            }
        }

        /// <summary>
        /// Registration methods.
        /// </summary>
        public enum RegistrationMethods
        {
            /// <summary>
            /// None
            /// </summary>
            None,

            /// <summary>
            /// Standard only.
            /// </summary>
            StandardOnly,

            /// <summary>
            /// Dependant only.
            /// </summary>
            DependantOnly,

            /// <summary>
            /// Standard or dependant.
            /// </summary>
            StandardOrDependant
        }

        /// <summary>
        /// Gets formatted Date of Birth.
        /// </summary>
        public string DateOfBirthFormatted
        {
            get { return this.DateOfBirth.HasValue ? this.DateOfBirth.Value.ToString("dd/MM/yyyy") : string.Empty; }
        }

        #endregion View Logic
    }
}