﻿using System;
using HIPS.PcehrSchemas;

namespace HIPS.Web.UI.ViewModels.DocumentManagement
{
    /// <summary>
    /// Represents a document that has been uploaded by this organisation to the PCEHR system.
    /// </summary>
    public class UploadedDocumentViewModel : DocumentManagementViewModelBase
    {
        #region Properties

        /// <summary>
        /// Gets or sets the admission date for the episode that the document relates to.
        /// </summary>
        public DateTime AdmissionDate { get; set; }

        /// <summary>
        /// Gets or sets the discharge date for the episode that the document relates to.
        /// </summary>
        public DateTime? DischargeDate { get; set; }

        /// <summary>
        /// Gets or sets the type of document.
        /// </summary>
        public string DocumentType { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider for the episode that the document relates to.
        /// </summary>
        public string ResponsibleProviderFamilyName { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider for the episode that the document relates to.
        /// </summary>
        public string ResponsibleProviderGivenNames { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider for the episode that the document relates to.
        /// </summary>
        public string ResponsibleProviderSuffix { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider for the episode that the document relates to.
        /// </summary>
        public string ResponsibleProviderTitle { get; set; }

        /// <summary>
        /// Gets or sets the document ID.
        /// </summary>
        public string SourceSystemDocumentId { get; set; }

        /// <summary>
        /// Gets or sets the document set ID.
        /// </summary>
        public string SourceSystemSetId { get; set; }

        /// <summary>
        /// Gets or sets the document status.
        /// </summary>
        public LocalDocumentStatus Status { get; set; }

        /// <summary>
        /// Gets or sets the date and time when the document was uploaded.
        /// </summary>
        public DateTime? UploadedDate { get; set; }

        /// <summary>
        /// Gets or sets the admission reason for the episode that the document relates to.
        /// </summary>
        public string AdmissionReason { get; set; }

        #endregion Properties

        #region Calculated Properties

        /// <summary>
        /// Gets the name of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProvider
        {
            get
            {
                return string.Format(
                    "{0} {1} {2} {3}",
                    this.ResponsibleProviderTitle,
                    this.ResponsibleProviderGivenNames,
                    this.ResponsibleProviderFamilyName,
                    this.ResponsibleProviderSuffix);
            }
        }

        /// <summary>
        /// Gets the document status as a string.
        /// </summary>
        public string DocumentStatus
        {
            get
            {
                switch (this.Status)
                {
                    case LocalDocumentStatus.Unknown:
                        return "Unknown";

                    case LocalDocumentStatus.PendingUpload:
                        return "Pending Upload";

                    case LocalDocumentStatus.Uploaded:
                        return "Uploaded";

                    case LocalDocumentStatus.FailedUpload:
                        return "Failed Upload";

                    case LocalDocumentStatus.PendingRemove:
                        return "Pending Remove";

                    case LocalDocumentStatus.Removed:
                        return "Removed";

                    case LocalDocumentStatus.FailedRemove:
                        return "Failed Remove";

                    default:
                        return "Unknown";
                }
            }
        }

        #endregion Calculated Properties
    }
}