using System;
using System.Collections.Generic;
using System.Linq;
using HIPS.CommonSchemas;
using HIPS.Web.Model.AssistedRegistration;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.AssistedRegistration;

namespace HIPS.Web.BusinessLogic.AssistedRegistration
{
    public class ValidatedAssistedRegistrationService : IAssistedRegistrationService
    {
        private IAssistedRegistrationService RegistrationService { get; set; }

        public ValidatedAssistedRegistrationService(IAssistedRegistrationService registrationService)
        {
            RegistrationService = registrationService;
        }

        private List<string> ValidateCommonRegistration(IndigenousStatus applicantIndigenousStatus, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod)
        {
            List<string> errors = new List<string>();

            // Validate Indigenous Status
            if (applicantIndigenousStatus == null)
            {
                errors.Add("Indigenous Status or 'Not Stated' must be selected.");
            }

            // Validate Identity Verification Method
            if (identityVerificationMethod == null)
            {
                errors.Add("Identity Verification Method is required.");
            }

            // Validate IVC Delivery Method selection
            if (ivcDeliveryMethod == null)
            {
                errors.Add("IVC Delivery Method selection is required.");
            }
            else
            {
                errors.AddRange(ivcDeliveryMethod.Validate());
            }

            return errors;
        }

        private List<string> ValidateRegistration(PersonDemographic applicant, Hospital applicantHospital, bool applicantConsent, IndigenousStatus applicantIndigenousStatus, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod)
        {
            List<string> errors = new List<string>();

            // Validate Applicant Consent
            if (!applicantConsent)
            {
                errors.Add("Applicant consent is required to process registration.");
            }

            // Validate Applicant
            errors.AddRange(ValidateIndependentApplicant(applicant));

            // Validate Hospital
            if (applicantHospital == null)
            {
                errors.Add("Applicant Hospital is required.");
            }

            // Validate Common Registration Properties
            errors.AddRange(ValidateCommonRegistration(applicantIndigenousStatus, medicareConsents, identityVerificationMethod, ivcDeliveryMethod));

            return errors;
        }

        private List<string> ValidateDependantRegistration(PersonDemographic applicant, Hospital applicantHospital, bool applicantConsent, IndigenousStatus applicantIndigenousStatus, PersonDemographic representative, bool representativeConsent, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod)
        {
            List<string> errors = new List<string>();

            // Validate Applicant and Representative Consent
            if (!applicantConsent || !representativeConsent)
            {
                errors.Add("Consent is required to process assisted registration.");
            }

            // Validate Applicant and Representative
            errors.AddRange(ValidateDependantApplicant(applicant, representative));

            // Validate Hospital
            if (applicantHospital == null)
            {
                errors.Add("Applicant Hospital is required.");
            }

            // Validate Common Registration Properties
            errors.AddRange(ValidateCommonRegistration(applicantIndigenousStatus, medicareConsents, identityVerificationMethod, ivcDeliveryMethod));

            return errors;
        }

        private static List<string> ValidateIndependentApplicant(PersonDemographic applicant)
        {
            List<string> errors = new List<string>();

            // Validate Applicant
            errors.AddRange(DemographicValidations.ValidateDemographic(applicant, false));

            // Ensure Applicant is 14 or older
            if (applicant.DateOfBirth > DateTime.Now.AddYears(-14))
            {
                errors.Add("Applicants aged under 14 years cannot be registered without a Representative.");
            }

            return errors;
        }

        private static List<string> ValidateDependantApplicant(PersonDemographic applicant, PersonDemographic representative)
        {
            List<string> errors = new List<string>();

            // Validate Applicant (Format errors)
            errors.AddRange(DemographicValidations.ValidateDemographic(applicant, false).Select(s => String.Format("Applicant: {0}", s)));

            // Validate Representative (Format errors)
            errors.AddRange(DemographicValidations.ValidateDemographic(representative, false).Select(s => String.Format("Representative: {0}", s)));

            if (applicant.DateOfBirth.HasValue && representative.DateOfBirth.HasValue)
            {
                // Ensure Applicant is not 18 or over
                if (applicant.DateOfBirth.Value <= DateTime.Now.AddYears(-18))
                {
                    errors.Add("Applicants aged 18 years or older cannot register a Representative.");
                }

                // Ensure Representative is 14 or older
                if (representative.DateOfBirth.Value > DateTime.Now.AddYears(-14))
                {
                    errors.Add("Representatives must be aged 14 years or older.");
                }

                // Ensure Representative is at least 14 years older than Applicant
                // NB: Assuming applicant isn't in future this is redundant?
                if (representative.DateOfBirth.Value > applicant.DateOfBirth.Value.AddYears(-14))
                {
                    errors.Add("Representatives must be 14 years older than Applicants.");
                }
            }

            //NB: Are there redundant checks here?
            // Ensure Representative are both using Medicare Numbers and that they are the same
            if (String.IsNullOrWhiteSpace(applicant.MedicareNumber) || String.IsNullOrWhiteSpace(representative.MedicareNumber) || applicant.MedicareNumber != representative.MedicareNumber)
            {
                errors.Add("The Applicant and Representative must share the same Medicare Number.");
            }

            return errors;
        }

        public void Dispose()
        {
            if (RegistrationService != null)
            {
                RegistrationService.Dispose();
                RegistrationService = null;
            }
        }

        public RegisterResponse RegisterApplicant(PersonDemographic applicant, Hospital applicantHospital, bool applicantConsent, IndigenousStatus applicantIndigenousStatus, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            // Validate
            List<string> validationErrors = ValidateRegistration(applicant, applicantHospital, applicantConsent, applicantIndigenousStatus, medicareConsents, identityVerificationMethod, ivcDeliveryMethod).ToList();
            if (validationErrors.Any())
            {
                return new RegisterResponse { Errors = validationErrors };
            }

            // Register
            return RegistrationService.RegisterApplicant(applicant, applicantHospital, applicantConsent, applicantIndigenousStatus, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        public RegisterResponse RegisterApplicant(ValidatedPatient applicant, bool applicantConsent, IndigenousStatus applicantIndigenousStatus, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            // Validate
            List<string> validationErrors = ValidateRegistration(applicant.RegisteredDemographics, applicant.CurrentLocation.Hospital, applicantConsent, applicantIndigenousStatus, medicareConsents, identityVerificationMethod, ivcDeliveryMethod).ToList();
            if (validationErrors.Any())
            {
                return new RegisterResponse { Errors = validationErrors };
            }

            // Register
            return RegistrationService.RegisterApplicant(applicant, applicantConsent, applicantIndigenousStatus, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        public RegisterResponse RegisterDependant(PersonDemographic applicant, Hospital applicantHospital, bool applicantConsent, IndigenousStatus applicantIndigenousStatus, PersonDemographic representative, bool representativeConsent, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            // Validate
            List<string> validationErrors = ValidateDependantRegistration(applicant, applicantHospital, applicantConsent, applicantIndigenousStatus, representative, representativeConsent, medicareConsents, identityVerificationMethod, ivcDeliveryMethod).ToList();
            if (validationErrors.Any())
            {
                return new RegisterResponse { Errors = validationErrors };
            }

            // Register
            return RegistrationService.RegisterDependant(applicant, applicantHospital, applicantConsent, applicantIndigenousStatus, representative, representativeConsent, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }

        public RegisterResponse RegisterDependant(ValidatedPatient applicant, bool applicantConsent, IndigenousStatus applicantIndigenousStatus, PersonDemographic representative, bool representativeConsent, Dictionary<MedicareConsent, bool> medicareConsents, IdentityVerificationMethod identityVerificationMethod, IvcDeliveryMethodDecision ivcDeliveryMethod, UserDetails operatingUser, int? applicantPatientMasterId)
        {
            // Validate
            List<string> validationErrors = ValidateDependantRegistration(applicant.RegisteredDemographics, applicant.CurrentLocation.Hospital, applicantConsent, applicantIndigenousStatus, representative, representativeConsent, medicareConsents, identityVerificationMethod, ivcDeliveryMethod).ToList();
            if (validationErrors.Any())
            {
                return new RegisterResponse { Errors = validationErrors };
            }

            // Register
            return RegistrationService.RegisterDependant(applicant, applicantConsent, applicantIndigenousStatus, representative, representativeConsent, medicareConsents, identityVerificationMethod, ivcDeliveryMethod, operatingUser, applicantPatientMasterId);
        }
    }
}