﻿using System;

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PatientSchemas;
using HIPS.Web.Components.ServiceModel;

namespace HIPS.Web.ModelInterface.Common
{
    /// <summary>
    /// Defines the operations to be supported by an implementation of a patient repository.
    /// </summary>
    public interface IPatientRepository : IDisposable
    {
        #region Methods

        /// <summary>
        /// Retrieves a list of patients who are currently admitted to (and have not been discharged from) a specified hospital.
        /// </summary>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PatientListResponse containing the results of the operation.</returns>
        ServiceResponse<PatientListResponse> ListPatientsCurrentlyInHospital(string hospitalCodeSystem, string hospitalCode, bool? withPcehr, CommonSchemas.UserDetails operatingUser);

        /// <summary>
        /// Retrieves a list of patients who are currently admitted or recently discharged from a specified hospital.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="hospitalCodeSystem">Code system used to identify the hospital.</param>
        /// <param name="hospitalCode">Code within the specified code system used to identify the hospital.</param>
        /// <param name="withIhi">Boolean value indicating whether to list patients with (true) or without (false) a verified active IHI. Null indicates all patients should be listed regardless of whether they have an IHI.</param>
        /// <param name="withPcehr">Boolean value indicating whether to list patients with (true) or without (false) a PCEHR. Null indicates all patients should be listed regardless of whether they have a PCEHR.</param>
        /// <param name="excludeMedicareExclusions">Boolean value indicating whether to exclude (true) or include (false) patients with a Medicare Care Number that exists in the HIPS MedicareExclusions table.</param>
        /// <returns>ServiceResponse containing AdmittedPatientListResponse containing the results of the operation.</returns>
        ServiceResponse<PatientSchemas.AdmittedPatientListResponse> ListAdmittedPatients(UserDetails user, string hospitalCodeSystem, string hospitalCode, bool? withIhi, bool? withPcehr, bool excludeMedicareExclusions);

        /// <summary>
        /// Retrieves the details associated with the disclosure of the existence of a PCEHR by a specified patient.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and the details associated with the disclosure.</returns>
        ServiceResponse<PatientDisclosureDetailsResponse> GetPatientDisclosureDetails(UserDetails user, Mrn patientIdentifer);

        /// <summary>
        /// Retrieves the details associated with the disclosure of the existence of a PCEHR by a specified patient.
        /// </summary>
        /// <param name="user">User requesting the operation.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and the details associated with the disclosure.</returns>
        ServiceResponse<PatientDisclosureDetailsResponse> GetPatientDisclosureDetails(UserDetails user, RegisteredEnterprisePatient patientIdentifer);

        #endregion Methods
    }
}