﻿using System;
using System.Collections.Generic;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.ServiceModel;

namespace HIPS.Web.ModelInterface.DocumentManagement
{
    /// <summary>
    /// Defines the operations to be supported by an implementation of an uploaded document repository.
    /// </summary>
    public interface IUploadedDocumentRepository : IDisposable
    {
        /// <summary>
        /// List the uploaded documents for a certain patient at a certain hospital.
        /// If setId is specified, lists all versions of the one document.
        /// If setId is not specified, lists the current version of all documents.
        /// </summary>
        /// <param name="user">Current user details.</param>
        /// <param name="setId">Identifier of the document set whose documents are to be listed.</param>
        /// <param name="patientIdentifier">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and list of documents.</returns>
        ServiceResponse<LocalClinicalDocumentListResponse<PatientIdentifierBase>> List(UserDetails user, string setId, Mrn patientIdentifier);

        /// <summary>
        /// Retrieve the contents of an uploaded document.
        /// </summary>
        /// <param name="user">Current user details.</param>
        /// <param name="setId">Identifier of the document set.</param>
        /// <param name="documentId">Identifier of the document instance.</param>
        /// <param name="patientIdentifer">Identifier of the patient and hospital.</param>
        /// <returns>Response containing status information and document contents.</returns>
        ServiceResponse<LocalClinicalDocumentResponse<PatientIdentifierBase>> Get(UserDetails user, string setId, string documentId, Mrn patientIdentifer);

        /// <summary>
        /// Enqueue a request for HIPS to remove an uploaded document from the PCEHR system.
        /// This marks the document set as 'removed' which makes it invisible to the patient
        /// and to other healthcare provider organisations. The status will be reset to
        /// 'uploaded' if a new version of the document is uploaded.
        /// </summary>
        /// <param name="patientIdentifier">Identifier of the patient and hospital.</param>
        /// <param name="admissionDate">Admission date and time used to identify the episode against which the document was uploaded.</param>
        /// <param name="setId">Identifier of the document set.</param>
        /// <param name="reason">Reason for removal: Withdrawn or IncorrectIdentity.</param>
        /// <param name="user">Current user details.</param>
        /// <param name="auditInformation">Reference notes entered by the user.</param>
        /// <returns>Response containing status information.</returns>
        ServiceResponse<HipsResponse> Remove(Mrn patientIdentifier, DateTime admissionDate, string setId, RemovalReason reason, UserDetails user, byte[] auditInformation);

        /// <summary>
        /// Cache a CdaDocumentDetails.
        /// </summary>
        /// <param name="setId">Identifier for the document.</param>
        /// <param name="cda">CdaDocumentDetails to be cached.</param>
        void SetCdaDocumentDetails(string setId, HIPS.Web.Model.Cda.CdaDocumentDetails cda);

        /// <summary>
        /// Gets CdaDocumentDetails from cache.
        /// </summary>
        /// <param name="setId">Identifier for the document.</param>
        /// <returns>A CdaDocumentDetails.</returns>
        HIPS.Web.Model.Cda.CdaDocumentDetails GetCdaDocumentDetails(string setId);

        /// <summary>
        /// Uploads a Level 3A discharge summary to the PCEHR National Repository,
        /// optionally superseding a previously-uploaded document. This method
        /// takes in the XML document and any attachments and does the packaging
        /// itself.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="cdaDocument">The CDA root XML document.</param>
        /// <param name="attachments">The attachments.</param>
        /// <param name="admissionDate">The admission date.</param>
        /// <param name="documentFormatCode">The document format code.</param>
        /// <param name="userContext">User requesting the operation.</param>
        /// <returns>Response containing status information.</returns>
        ServiceResponse<HipsResponse> UploadOrSupersedeDocument(HIPS.Web.Model.PatientIdentifier.Mrn patientIdentifier, byte[] cdaDocument, List<HIPS.Web.Model.Cda.CdaAttachment> attachments, DateTime admissionDate, string documentFormatCode, UserDetails userContext);
    }
}