﻿using System;
using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrSchemas;
using HIPS.Web.Components.ServiceModel;

namespace HIPS.Web.ModelInterface.PcehrView
{
    /// <summary>
    /// Defines the operations to be supported by an implementation of a PCEHR viewing repository.
    /// </summary>
    public interface IPcehrViewRepository : IDisposable
    {
        #region Methods

        /// <summary>
        /// Retrieves a list of active documents for a specified patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DocumentListResponse containing the results of the operation.</returns>
        ServiceResponse<DocumentListResponse<PatientIdentifierBase>> ListActiveDocuments(RegisteredEnterprisePatient patientIdentifier, UserDetails operatingUser);

        /// <summary>
        /// Retrieves a list of active documents for a specified patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DocumentListResponse containing the results of the operation.</returns>
        ServiceResponse<DocumentListResponse<PatientIdentifierBase>> ListActiveDocuments(Mrn patientIdentifier, UserDetails operatingUser);

        /// <summary>
        /// Retrieves a specific document for a patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <param name="documentSpec">Identifier for the document.</param>
        /// <returns>DocumentResponse containing the results of the operation.</returns>
        ServiceResponse<DocumentResponse> GetDocument(Mrn patientIdentifier, UserDetails operatingUser, DocumentRequest documentSpec);

        /// <summary>
        /// Retrieves a specific view for a patient.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <param name="viewRequest">Criteria for the view.</param>
        /// <returns>ViewResponse containing the requested view.</returns>
        ServiceResponse<ViewResponse> GetView(Mrn patientIdentifier, UserDetails operatingUser, PrescriptionAndDispenseViewRequest viewRequest);

        /// <summary>
        /// Retrieves details of the patient's PCEHR status.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="dateOfBirth">Date of birth for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>DoesPcehrExistResponse containing the results of the operation.</returns>
        ServiceResponse<DoesPcehrExistResponse> IsPcehrAdvertised(Mrn patientIdentifier, DateTime dateOfBirth, UserDetails operatingUser);

        /// <summary>
        /// Attempts to gain access to a patient's PCEHR without using an access code.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        ServiceResponse<GainPcehrAccessResponse> GainAccessWithoutCode(Mrn patientIdentifier, UserDetails operatingUser);

        /// <summary>
        /// Attempts to gain access to a patient's PCEHR using an access code.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="accessCode">Access code provided by the user.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        ServiceResponse<GainPcehrAccessResponse> GainAccessWithCode(Mrn patientIdentifier, string accessCode, UserDetails operatingUser);

        /// <summary>
        /// Attempts to gain access emergency to a patient's PCEHR.
        /// </summary>
        /// <param name="patientIdentifier">Identifier for the patient.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>GainPcehrAccessResponse containing the results of the operation.</returns>
        ServiceResponse<GainPcehrAccessResponse> GainAccessEmergency(Mrn patientIdentifier, UserDetails operatingUser);

        #endregion Methods
    }
}