using System;
using System.Collections.Generic;
using System.Web;
using System.Web.Mvc;
using HIPS.Web.BusinessLogic.AssistedRegistration;
using HIPS.Web.Components.Cache;
using HIPS.Web.Data.Hips;
using HIPS.Web.Data.Hips.ConsentManagement;
using HIPS.Web.Data.Hips.DocumentManagement;
using HIPS.Web.Data.Hips.PcehrView;
using HIPS.Web.Data.Hips.Reference;
using HIPS.Web.Data.WebsiteDb;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.AssistedRegistration;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.ConsentManagement;
using HIPS.Web.ModelInterface.DischargeSummary;
using HIPS.Web.ModelInterface.DocumentManagement;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.UI.App_Start;
using HIPS.Web.UI.Controllers;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Properties;
using Microsoft.Web.Infrastructure.DynamicModuleHelper;
using Ninject;
using Ninject.Web.Common;
using Ninject.Web.Mvc.FilterBindingSyntax;
using WebActivator;

[assembly: WebActivator.PreApplicationStartMethod(typeof(NinjectWebCommon), "Start")]
[assembly: ApplicationShutdownMethod(typeof(NinjectWebCommon), "Stop")]

namespace HIPS.Web.UI.App_Start
{
    /// <summary>
    /// Start-up module for Ninject.
    /// </summary>
    public static class NinjectWebCommon
    {
        #region Fields

        /// <summary>
        /// Object used to set up the Ninject bindings.
        /// </summary>
        private static readonly Bootstrapper Bootstrapper = new Bootstrapper();

        #endregion Fields

        #region Methods

        /// <summary>
        /// Starts the application
        /// </summary>
        public static void Start()
        {
            DynamicModuleUtility.RegisterModule(typeof(OnePerRequestHttpModule));
            DynamicModuleUtility.RegisterModule(typeof(NinjectHttpModule));
            Bootstrapper.Initialize(CreateKernel);

        }

        /// <summary>
        /// Stops the application.
        /// </summary>
        public static void Stop()
        {
            Bootstrapper.ShutDown();
        }

        /// <summary>
        /// Creates the kernel that will manage your application.
        /// </summary>
        /// <returns>The created kernel.</returns>
        private static IKernel CreateKernel()
        {
            var kernel = new StandardKernel();
            kernel.Bind<Func<IKernel>>().ToMethod(ctx => () => new Bootstrapper().Kernel);
            kernel.Bind<IHttpModule>().To<HttpApplicationInitializationHttpModule>();

            RegisterServices(kernel);
            return kernel;
        }

        /// <summary>
        /// Load your modules or register your services here!
        /// </summary>
        /// <param name="kernel">The kernel.</param>
        private static void RegisterServices(IKernel kernel)
        {
            //--- SettingsRepository ---//

            /* When asked to provide an ISettingsRepository use a CachedSettingsRepository */
            kernel.Bind<ISettingsRepository>()
                .To<CachedSettingsRepository>();

            /* When creating a CachedSettingsRepository use the MemoryCache for caching  */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedSettingsRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.SettingsRepository_AbsoluteExpirationOffset);

            /* When creating a CachedSettingsRepository use a WebsiteDbRepository as the source ISettingsRepository */
            kernel.Bind<ISettingsRepository>()
                .To<WebsiteDbRepository>()
                .WhenInjectedInto<CachedSettingsRepository>();

            //--- HospitalRepository ---//

            /* When asked to provide an IHospitalRepository use a CachedHospitalRepository */
            kernel.Bind<IHospitalRepository>()
                .To<CachedHospitalRepository>();

            /* When creating a CachedHospitalRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedHospitalRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.HospitalRepository_AbsoluteExpirationOffset);

            /* When creating a CachedHospitalRepository use a HospitalRepository as the source IHospitalRepository */
            kernel.Bind<IHospitalRepository>()
               .To<HospitalRepository>()
               .WhenInjectedInto<CachedHospitalRepository>();

            //--- AssistedRegistrationReferenceRepository ---//

            /* When asked to provide an IAssistedRegistrationReferenceRepository use a CachedAssistedRegistrationReferenceRepository */
            kernel.Bind<IAssistedRegistrationReferenceRepository>()
                .To<CachedAssistedRegistrationReferenceRepository>();

            /* When creating a CachedAssistedRegistrationReferenceRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedAssistedRegistrationReferenceRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.AssistedRegistrationReferenceRepository_AbsoluteExpirationOffset);

            /* When creating a CachedAssistedRegistrationReferenceRepository use a WebsiteDbRepository as the source IAssistedRegistrationReferenceRepository */
            kernel.Bind<IAssistedRegistrationReferenceRepository>()
                .To<WebsiteDbRepository>()
                .WhenInjectedInto<CachedAssistedRegistrationReferenceRepository>();

            //--- DischargeSummaryReferenceRepository ---//

            /* When asked to provide an IDischargeSummaryReferenceRepository use a CachedDischargeSummaryReferenceRepository */
            kernel.Bind<IDischargeSummaryReferenceRepository>()
                .To<CachedDischargeSummaryReferenceRepository>();

            /* When creating a CachedDischargeSummaryReferenceRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<CachedDischargeSummaryReferenceRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.DischargeSummaryReferenceRepository_AbsoluteExpirationOffset);

            /* When creating a CachedDischargeSummaryReferenceRepository use a WebsiteDbRepository as the source IDischargeSummaryReferenceRepository */
            kernel.Bind<IDischargeSummaryReferenceRepository>()
                .To<WebsiteDbRepository>()
                .WhenInjectedInto<CachedDischargeSummaryReferenceRepository>();

            //--- AssistedRegistrationService ---//

            /* When asked to provide an IAssistedRegistrationService use a ValidatedAssistedRegistrationService */
            kernel.Bind<IAssistedRegistrationService>()
               .To<ValidatedAssistedRegistrationService>();

            /* When creating a ValidatedAssistedRegistrationService use an AssistedRegistrationService as the source IAssistedRegistrationService */
            kernel.Bind<IAssistedRegistrationService>()
                .To<AssistedRegistrationService>()
                .WhenInjectedInto<ValidatedAssistedRegistrationService>();

            //--- ConsentManagementService ---//

            /* When asked to provide an IConsentManagementService use a ConsentManagementService */
            kernel.Bind<IConsentManagementService>()
               .To<ConsentManagementService>();

            //--- PatientRepository ---//
            // Need to retrieve the days discharged from settings so we can pass in to constructor.
            Dictionary<Setting.SettingCodes, int> daysDischarged = new Dictionary<Setting.SettingCodes, int>();
            List<Setting.SettingCodes> daysDischargedSettings = new List<Setting.SettingCodes>()
            {
                Setting.SettingCodes.PcehrViewPatientDaysDischarged,
                Setting.SettingCodes.PatientsWithoutIhiDaysDischarged,
                Setting.SettingCodes.DiscloseHiddenPcehrDaysDischarged,
                Setting.SettingCodes.WithdrawConsentPatientListDaysDischarged,
                Setting.SettingCodes.WithdrawConsentEpisodeListDaysDischarged,
                Setting.SettingCodes.RemoveDocumentDaysDischarged,
                Setting.SettingCodes.DischargeSummaryPatientListDaysDischarged,
                Setting.SettingCodes.DischargeSummaryEpisodeListDaysDischarged
            };
            using (WebsiteDbRepository settings = new WebsiteDbRepository())
            {
                foreach (var code in daysDischargedSettings)
                {
                    daysDischarged[code] = int.Parse(settings.GetSettings().GetSettingValue(code));
                }
            }

            /* When asked to provide an IPatientRepository use a PatientRepository */
            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<PcehrViewController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.PcehrViewPatientDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<PcehrAccessFilter>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.PcehrViewPatientDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<PcehrAccessManager>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.PcehrViewPatientDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<AssistedRegistrationController>()
                .WithConstructorArgument("daysDischarged", 0);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<DataIntegrityController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.PatientsWithoutIhiDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<ConsentManagementController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.WithdrawConsentPatientListDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<DisclosureManagementController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.DiscloseHiddenPcehrDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<DocumentManagementController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.RemoveDocumentDaysDischarged]);

            kernel.Bind<IPatientRepository>()
                .To<PatientRepository>()
                .WhenInjectedInto<DocumentUploadController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.DischargeSummaryPatientListDaysDischarged]);

            /* When creating a PatientRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<PatientRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.PatientRepository_AbsoluteExpirationOffset);

            //--- EpisodeRepository ---//

            /* When asked to provide an IEpisodeRepository use a EpisodeRepository */
            kernel.Bind<IEpisodeRepository>()
                .To<EpisodeRepository>()
                .WhenInjectedInto<ConsentManagementController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.WithdrawConsentEpisodeListDaysDischarged]);

            kernel.Bind<IEpisodeRepository>()
                .To<EpisodeRepository>()
                .WhenInjectedInto<DocumentUploadController>()
                .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.DischargeSummaryEpisodeListDaysDischarged]);

            /* When asked to provide an IEpisodeRepositoryV2 use a EpisodeRepositoryV2 */
            kernel.Bind<IEpisodeRepositoryV2>()
               .To<EpisodeRepositoryV2>()
               .WhenInjectedInto<DocumentUploadController>()
               .WithConstructorArgument("daysDischarged", daysDischarged[Setting.SettingCodes.DischargeSummaryEpisodeListDaysDischarged]);

            //--- PcehrViewRepository ---//

            /* When asked to provide an IPcehrViewRepository use a PcehrViewRepository */
            kernel.Bind<IPcehrViewRepository>()
                .To<PcehrViewRepository>();

            /* When creating a PcehrViewRepository use the MemoryCache for caching */
            kernel.Bind<ICacheProvider>()
                .To<MemoryCacheProvider>()
                .WhenInjectedInto<PcehrViewRepository>()
                .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.PcehrViewRepository_AbsoluteExpirationOffset);

            //--- PcehrAccessFilter ---//

            /* When an action method has the PcehrAccessAttribute apply the PcehrAccessFilter */
            kernel.BindFilter<PcehrAccessFilter>(FilterScope.Action, 0)
                .WhenActionMethodHas<PcehrAccessAttribute>()
                .WithConstructorArgumentFromActionAttribute<PcehrAccessAttribute>("autoAccess", a => a.AutoAccess);

            //--- UploadedDocumentRepository ---//

            kernel.Bind<IUploadedDocumentRepository>()
                .To<UploadedDocumentRepository>();

            kernel.Bind<ICacheProvider>()
               .To<MemoryCacheProvider>()
               .WhenInjectedInto<UploadedDocumentRepository>()
               .WithConstructorArgument("defaultAbsoluteExpirationOffset", Settings.Default.PatientRepository_AbsoluteExpirationOffset);

            //--- CdaRepository ---//

            /* When asked to provide an ICdaRepository use a CdaRepository */
            kernel.Bind<ICdaRepository>()
              .To<CdaRepository>()
              .WhenInjectedInto<DocumentUploadController>();

            //--- SessionConfiguration ---//

            /* When asked to provide an ISessionProvider use an HttpSessionProvider */
            kernel.Bind<ISessionProvider>()
                .To<HIPS.Web.UI.Helpers.HttpSessionProvider>();

            /* When asked to provide an ISessionConfiguration, use a SessionConfiguration */
            kernel.Bind<ISessionConfiguration>()
                .To<HIPS.Web.UI.Helpers.SessionConfiguration>();

            //--- HpoRequiredFilter ---//

            /* When a controller has the HpoRequiredAttribute apply the HpoRequiredFilter */
            kernel.BindFilter<HpoRequiredFilter>(FilterScope.Controller, 0)
                .WhenControllerHas<HpoRequiredAttribute>();

            /* When an action method has the HpoRequiredAttribute apply the HpoRequiredFilter */
            kernel.BindFilter<HpoRequiredFilter>(FilterScope.Action, 0)
                .WhenActionMethodHas<HpoRequiredAttribute>();
        }

        #endregion Methods
    }
}