﻿// <copyright file="AccountController.Windows.cs" auther="Mohammad Younes">
// Copyright 2013 Mohammad Younes.
// 
// Released under the MIT license
// http://opensource.org/licenses/MIT
//
// </copyright>

using System;
using System.Collections.Generic;
using System.Linq;
using System.Security.Claims;
using System.Threading.Tasks;
using System.Web;
using System.Web.Mvc;
using Microsoft.AspNet.Identity;
using Microsoft.AspNet.Identity.EntityFramework;
using Microsoft.Owin.Security;
using HIPS.Web.UI.ViewModels.Account;
using HIPS.Web.UI.Helpers;
using System.Security.Principal;
using System.DirectoryServices.AccountManagement;
using Microsoft.AspNet.Identity.Owin;
using System.DirectoryServices;
using System.Web.UI;
using System.Net.Http;
using System.Net;
using System.IO;
using System.Drawing;
using System.Drawing.Imaging;

namespace HIPS.Web.UI.Controllers
{
    [Authorize]
    public partial class AccountController : Controller
    {
        //
        // POST: /Account/WindowsLogin
        [AllowAnonymous]
        [ValidateAntiForgeryToken]
        [HttpPost]
        [OutputCache(NoStore = true, Location = OutputCacheLocation.None)]
        public async Task<ActionResult> WindowsLogin(string userName, string returnUrl)
        {
            if (!Request.LogonUserIdentity.IsAuthenticated)
            {
                return RedirectToAction("Login");
            }

            var loginInfo = GetWindowsLoginInfo();

            // Sign in the user with this external login provider if the user already has a login
            var user = await UserManager.FindAsync(loginInfo);
            if (user != null)
            {
                // Get the user's AD groups and add them to the Claims Role
                MapWindowsIdentityToAppUserClaims(user, Request.LogonUserIdentity);
                await SignInAsync(user, isPersistent: false);
                return RedirectToLocal(returnUrl);
            }
            else
            {
                // If the user does not have an account, then prompt the user to create an account
                var name = userName;
                if (string.IsNullOrEmpty(name))
                    name = Request.LogonUserIdentity.Name.Split('\\')[1];
                var appUser = new ApplicationUser() { UserName = name };
                var result = await UserManager.CreateAsync(appUser);
                if (result.Succeeded)
                {
                    result = await UserManager.AddLoginAsync(appUser.Id, loginInfo);
                    if (result.Succeeded)
                    {
                        MapWindowsIdentityToAppUserClaims(appUser, Request.LogonUserIdentity);
                        await SignInAsync(appUser, isPersistent: false);
                        return RedirectToLocal(returnUrl);
                    }
                }
                AddErrors(result);
                ViewBag.ReturnUrl = returnUrl;
                ViewBag.LoginProvider = "Windows";
                return View("WindowsLoginConfirmation", new WindowsLoginConfirmationViewModel { UserName = name });
            }
        }

        //
        // POST: /Account/WindowsLogOff
        [HttpPost]
        [ValidateAntiForgeryToken]
        [OutputCache(NoStore = true, Location = OutputCacheLocation.None)]
        public void WindowsLogOff()
        {
            AuthenticationManager.SignOut();
            
        }

        //
        // POST: /Account/LinkWindowsLogin
        [AllowAnonymous]
        [HttpPost]
        public async Task<ActionResult> LinkWindowsLogin()
        {
            string userId = HttpContext.ReadUserId();

            //didn't get here through handler
            if (string.IsNullOrEmpty(userId))
                return RedirectToAction("Login");

            HttpContext.Items.Remove("windows.userId");

            //not authenticated.
            var loginInfo = GetWindowsLoginInfo();
            if (loginInfo == null)
                return RedirectToAction("Manage");

            //add linked login
            var result = await UserManager.AddLoginAsync(userId, loginInfo);

            //sign the user back in.
            var user = await UserManager.FindByIdAsync(userId);
            if (user != null)
                await SignInAsync(user, false);

            if (result.Succeeded)
                return RedirectToAction("Manage");

            return RedirectToAction("Manage", new { Message = ManageMessageId.Error });
        }

        /// <summary>
        /// Allows a user to enter AD details into the form to login
        /// </summary>
        /// <param name="userName">username entered on form</param>
        /// <param name="password">password entered on form</param>
        /// <param name="isPersistent">if Remeber Me is selected</param>
        /// <param name="returnUrl">return Url is successful</param>
        /// <returns>an action result, either allows user to continue or returns to Login Url</returns>
        [AllowAnonymous]
        [ValidateAntiForgeryToken]
        [HttpPost]
        [OutputCache(NoStore = true, Location = OutputCacheLocation.None)]
        public async Task<ActionResult> WindowsFormLogin(string userName, string password, bool isPersistent, string returnUrl)
        {
            // Get the context type from web config
            ContextType ct = GetConfiguredContextType();
            
            // Validate user's credentials
            using (var context = new PrincipalContext(ct, System.Configuration.ConfigurationManager.AppSettings["AccountManagement.DomainName"].ToString()))    
            {
                // validate the credentials
                bool credentialsValid = context.ValidateCredentials(userName, password);

                if (credentialsValid)
                {
                    UserPrincipal userPrincipal = UserPrincipal.FindByIdentity(context, userName);

                    if (userPrincipal != null)
                    {
                        var loginInfo = new UserLoginInfo("Windows", userPrincipal.Sid.ToString());
                        string domain = userName.Contains("\\") ? userName.Split('\\')[0] : "";
                        
                        var user = await UserManager.FindAsync(loginInfo);
                        if (user != null)
                        {
                            // Get the user's AD groups and add them to the Claims Role
                            AddDomainToRoleClaims(user, domain);
                            MapUserPrincipalToAppUserClaims(user, userPrincipal);
                            await SignInAsync(user, isPersistent);
                            return RedirectToLocal(returnUrl);
                        }
                        else
                        {
                            // If the user does not have an account, then prompt the user to create an account
                            var name = userName;
                            if (string.IsNullOrEmpty(name))
                            {
                                ModelState.AddModelError("", "Invalid username or password.");
                                return RedirectToAction("Login");
                            }
                            var appUser = new ApplicationUser() { UserName = name };
                            var result = await UserManager.CreateAsync(appUser);
                            if (result.Succeeded)
                            {
                                result = await UserManager.AddLoginAsync(appUser.Id, loginInfo);
                                if (result.Succeeded)
                                {
                                    // Get the user's AD groups and add them to the Claims Role
                                    AddDomainToRoleClaims(user, domain);
                                    MapUserPrincipalToAppUserClaims(appUser, userPrincipal);

                                    await SignInAsync(appUser, isPersistent: false);
                                    return RedirectToLocal(returnUrl);
                                }
                            }
                            AddErrors(result);
                            ViewBag.ReturnUrl = returnUrl;
                            ViewBag.LoginProvider = "Windows";
                            return View("WindowsLoginConfirmation", new WindowsLoginConfirmationViewModel { UserName = name });
                        }
                       
                    }
                }
                ModelState.AddModelError("", "Invalid username or password.");
                ViewBag.ReturnUrl = returnUrl;
                return RedirectToAction("Login");
            }
            
        }
        #region helpers
        private UserLoginInfo GetWindowsLoginInfo()
        {
            if (!Request.LogonUserIdentity.IsAuthenticated)
                return null;

            return new UserLoginInfo("Windows", Request.LogonUserIdentity.User.ToString());
        }
        /// <summary>
        /// Gets the Windows Identity AD groups from the Claims collection and enumrates to add to the ApplciationUser object
        /// </summary>
        /// <param name="user">Application User object</param>
        /// <param name="windowsIdentity">Logged on Windows Identity</param>
        private void MapWindowsIdentityToAppUserClaims(ApplicationUser user, WindowsIdentity windowsIdentity)
        {
            string domain = windowsIdentity.Name.Contains("\\") ? windowsIdentity.Name.Split('\\')[0] : "";
            AddDomainToRoleClaims(user, domain);
            
            // Get the context type from web config
            ContextType ct = GetConfiguredContextType();
            using (var context = new PrincipalContext(ct, System.Configuration.ConfigurationManager.AppSettings["AccountManagement.DomainName"].ToString()))
            {
                UserPrincipal userPrincipal = UserPrincipal.FindByIdentity(context, user.UserName);

                if (userPrincipal != null)
                {
                    MapUserPrincipalToAppUserClaims(user, userPrincipal);
                    DownloadADProfileImage(user, userPrincipal);
                }
                else
                {
                    bool blnGotGivenName = false;
                    bool blnGotSurname = false;

                    foreach (var claim in windowsIdentity.Claims)
                    {
                         switch (claim.Type)
                         {
                            case ClaimTypes.GroupSid:
                                        string groupName;
                                        try
                                        {
                                            groupName = new SecurityIdentifier(claim.Value).Translate(typeof(NTAccount)).ToString();
                                        }
                                        catch
                                        {
                                            groupName = claim.Value;
                                        }
                                 string groupDomain = groupName.Contains("\\") ? groupName.Split('\\')[0] : "";
                                 groupName = groupName.Replace(groupDomain + "\\", string.Empty);
                                 AddGroupNameToRoleClaims(user, groupName);
                                 break;
                             case ClaimTypes.GivenName:
                                 AddGivenNameToRoleClaims(user, claim.Value);
                                 blnGotGivenName = true;
                                 break;
                             case ClaimTypes.Surname:
                                 AddSurnameToRoleClaims(user, claim.Value);
                                 blnGotSurname = true;
                                 break;
                         }
                     }
                    // if we didnt get a givenName or Surname then fall back on login name
                    string fullName = user.UserName.Replace('.', ' ');
                    if (!blnGotGivenName)
                        AddGivenNameToRoleClaims(user, fullName.Split(' ').First());
                    if (!blnGotSurname)
                        AddSurnameToRoleClaims(user, fullName.Split(' ').Last());

                    DownloadADProfileImage(user, UserPrincipal.Current);
                }
            }   
        }
        /// <summary>
        /// Enumerates the UserPrincipal group collection and adds them to the Application User object
        /// </summary>
        /// <param name="user">ApplicationUser object</param>
        /// <param name="userPrincipal">UserPrincipal object</param>
        private void MapUserPrincipalToAppUserClaims(ApplicationUser user, UserPrincipal userPrincipal)
        {
            foreach (var group in userPrincipal.GetGroups())
            {
                AddGroupNameToRoleClaims(user, group.Name);
            }

            // Try to get the full name from the AD
            bool gotFullName = false;
            if (userPrincipal.GivenName != null && userPrincipal.Surname != null)
            {
                AddGivenNameToRoleClaims(user, userPrincipal.GivenName);
                AddSurnameToRoleClaims(user, userPrincipal.Surname);
                gotFullName = true;
            }
            else if (userPrincipal.GetUnderlyingObject() is DirectoryEntry)
            {
                // Try to extract the user's name from the Full Name field instead.
                DirectoryEntry entry = userPrincipal.GetUnderlyingObject() as DirectoryEntry;
                if (entry.Properties.Contains("FullName"))
                {
                    string fullName = entry.Properties["FullName"].Value.ToString();
                    if (!string.IsNullOrEmpty(fullName))
                    {
                        AddGivenNameToRoleClaims(user, fullName.Split(' ').First());
                        AddSurnameToRoleClaims(user, fullName.Split(' ').Last());
                        gotFullName = true;
                    }
                }
            }

            // Fall back to account name if the full name has not been found.
            if (!gotFullName)
            {
                string fullName = userPrincipal.SamAccountName.Replace('.', ' ');
                AddGivenNameToRoleClaims(user, fullName.Split(' ').First());
                AddSurnameToRoleClaims(user, fullName.Split(' ').Last());
            }

            DownloadADProfileImage(user, userPrincipal);
        }
        /// <summary>
        /// Adds the AD Group Name to the ApplicationUser object.
        /// </summary>
        /// <param name="user">ApplicationUser object to add claim to</param>
        /// <param name="groupName">AD Group Name to be added</param>
        private void AddGroupNameToRoleClaims(ApplicationUser user, string groupName)
        {
                user.Claims.Add(
                    new IdentityUserClaim()
                    {
                        ClaimType = ClaimTypes.Role,
                        ClaimValue = groupName 
                    });
        }
        private void AddDomainToRoleClaims(ApplicationUser user, string userDomain)
        {
            user.Claims.Add(
                new IdentityUserClaim()
                {
                    ClaimType = "userDomain",
                    ClaimValue = (userDomain != string.Empty ? userDomain : System.Configuration.ConfigurationManager.AppSettings["AccountManagement.DomainName"].ToString())
                });
        }
        private void AddGivenNameToRoleClaims(ApplicationUser user, string givenName)
        {
            user.Claims.Add(
                new IdentityUserClaim()
                {
                    ClaimType = ClaimTypes.GivenName,
                    ClaimValue = givenName
                });
        }
        private void AddSurnameToRoleClaims(ApplicationUser user, string surname)
        {
            user.Claims.Add(
                new IdentityUserClaim()
                {
                    ClaimType = ClaimTypes.Surname,
                    ClaimValue = surname
                });
        }
        private void DownloadADProfileImage(ApplicationUser user, UserPrincipal userPrincipal)
        {
            string defaultUri = string.Format("{0}{1}", System.Configuration.ConfigurationManager.AppSettings["Account.ProfileImageFilePath"].ToString(), System.Configuration.ConfigurationManager.AppSettings["Account.ProfileImageUriDefault"].ToString());
            string uri = defaultUri;
            string filename = Server.MapPath(uri);

            // Get the context type from web config
            ContextType ct = GetConfiguredContextType();
            byte[] profileImg = null;
            if (ct == ContextType.Machine)
            {
                // try and load the local temp one
                try
                {
                    profileImg = System.IO.File.ReadAllBytes(string.Format("{0}Users\\{1}\\AppData\\Local\\Temp\\{1}.bmp", Path.GetPathRoot(Environment.GetFolderPath(Environment.SpecialFolder.UserProfile)), user.UserName));
                }
                catch
                {
                    profileImg = null;
                }
            }
            else // get from AD
            {
                DirectoryEntry dirEntry = (DirectoryEntry)userPrincipal.GetUnderlyingObject();
                PropertyValueCollection col = dirEntry.Properties[System.Configuration.ConfigurationManager.AppSettings["Account.ProfileImageProperty"].ToString()];
                if (col.Value != null && col.Value is byte[])
                    profileImg = (byte[])col.Value;
            }
            
            if (profileImg != null && profileImg is byte[])
            {
                uri = string.Format("{0}{1}.jpg", System.Configuration.ConfigurationManager.AppSettings["Account.ProfileImageFilePath"].ToString(), user.Id.ToString());
                filename = Server.MapPath(uri);
                try
                {
                    // Delete the file if it exists already
                    if (System.IO.File.Exists(filename))
                        System.IO.File.Delete(filename);

                    byte[] thumbnailInBytes = (byte[])profileImg;
                    System.Drawing.ImageConverter converter = new System.Drawing.ImageConverter();
                    using (Image img = (Image)converter.ConvertFrom(thumbnailInBytes))
                    {
                        img.Save(filename, ImageFormat.Jpeg);
                    }
                }
                catch
                {
                    uri = defaultUri;
                }
            }
            user.Claims.Add(
                new IdentityUserClaim()
                {
                    ClaimType = "profileImageUri",
                    ClaimValue = uri
                });
        }
        
        private static ContextType GetConfiguredContextType()
        {
            // Get the context type from web config
            ContextType ct;
            try
            {
                ct = (ContextType)Enum.Parse(typeof(ContextType), System.Configuration.ConfigurationManager.AppSettings["AccountManagement.ContextType"].ToString());
            }
            catch
            {
                ct = ContextType.Domain;
            }
            return ct;
        }
        #endregion
    }
}