#region Using Statements

using System;
using System.Collections.Generic;
using System.DirectoryServices.AccountManagement;
using System.Linq;
using System.Security.Principal;
using System.Web.Mvc;
using HIPS.Web.Model.Common;
using HIPS.Web.Model.Identity;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.UI.Helpers;
using System.Security.Claims;

#endregion Using Statements

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Base class for controllers.
    /// </summary>
    public abstract class ControllerBase : Controller
    {
        #region Fields

        /// <summary>
        /// List of application-specific settings.
        /// </summary>
        protected readonly List<Setting> Settings;

        /// <summary>
        /// Gets or sets the session configuration used by this controller.
        /// </summary>
        protected readonly ISessionConfiguration SessionConfiguration;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="ControllerBase"/> class.
        /// </summary>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public ControllerBase(ISettingsRepository settingsRepository, ISessionConfiguration sessionConfiguration)
        {
            this.SessionConfiguration = sessionConfiguration;
            using (settingsRepository)
            {
                this.Settings = settingsRepository.GetSettings();
            }
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets the default hospital code system used by the application.
        /// </summary>
        protected string DefaultHospitalCodeSystem
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.DefaultHospitalCodeSystem);
            }
        }


        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets the local user details.
        /// </summary>
        /// <returns>The local user.</returns>
        protected LocalUser GetLocalUser()
        {
            ClaimsIdentity cid = null;

            // Attempt to get identity from controller.
            if ((this.User != null) && (this.User.Identity != null))
            {
                // Claims identity created at login
                cid = (ClaimsIdentity)this.User.Identity;
            }

            //If controller identity not present, fall-back to thread identity.
            if (cid == null)
            {
                cid = (ClaimsIdentity)System.Threading.Thread.CurrentPrincipal.Identity;
            }

            if (!cid.IsAuthenticated)
            {
                throw new InvalidOperationException("User login is required."); // TODO: Handle this differently?
            }

            string domain = cid.FindFirst("userDomain").Value;
            string loginName = cid.Name.Replace(domain + "\\", string.Empty);
            string familyName = (cid.FindFirst(ClaimTypes.Surname) != null) ? cid.FindFirst(ClaimTypes.Surname).Value : loginName; // Default to login.
            string givenNames = (cid.FindFirst(ClaimTypes.GivenName) != null) ? cid.FindFirst(ClaimTypes.GivenName).Value : "";
            
            string accessingHpio = string.Empty;
            if (this.SessionConfiguration.RepresentingHospital != null)
            {
                accessingHpio = this.SessionConfiguration.RepresentingHospital.Hpio;
            }

            return new LocalUser()
            {
                AccessingHpio = accessingHpio,
                Domain = domain,
                Login = loginName,
                GivenNames = givenNames,
                FamilyName = familyName
            };
        }

        #endregion Methods
    }
}