#region Using Statements

using System.Collections.Generic;
using System.Linq;
using System.Web.Mvc;

using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.Common;
using HIPS.Web.UI.ViewModels.Shared;
using System.Security.Claims;
using System.Web;
using System.Net;

#endregion Using Statements

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for common features.
    /// </summary>
    public class CommonController : HIPS.Web.UI.Controllers.ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="CommonController"/> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public CommonController(IHospitalRepository hospitalRepository, ISettingsRepository settingsRepository, ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
        }

        #endregion Constructors

        #region Methods

        #region Actions

        #region Session Information

        /// <summary>
        /// Display session information.
        /// </summary>
        /// <returns>Partial view for session information.</returns>
        [ChildActionOnly]
        public ActionResult SessionInfo()
        {
            var m = new SessionInfoViewModel();
            if (this.User.Identity.IsAuthenticated)
            { 
                m.User = this.GetCurrentUserDetails().Name;
                m.UserProfileImageUri = this.GetCurrentUserProfileImageUri();

                if (this.SessionConfiguration.RepresentingHospital == null)
                {
                    IEnumerable<HospitalViewModel> hospitals = SetHospitalForUser(); // see if this user has only 1 hospital assigned to them and default to that one
                    m.NumberHpoAccessible = (hospitals != null ? hospitals.Count() : 0);

                    // if the user has access to only 1 hospital then set the Session object for RepresentingHospital
                    if (hospitals != null && hospitals.Count() == 1)
                    {
                        var selectedHospital = hospitals.First();
                        if (selectedHospital != null)
                            this.SessionConfiguration.RepresentingHospital = ObjectMapper.Map<Hospital>(selectedHospital);
                    }
                    // if the user does not have access
                    if (hospitals == null || hospitals.Count() == 0)
                    {
                        // set the Hpo to the error message
                        m.Hpo = "None";
                        m.NumberHpoAccessible = 0;
                        
                        if (Request.QueryString["NumberHpoAccessible"] == null)
                        {
                            // Handle error
                            ViewBag.Error = "You have not been granted access to a Hpo. ";
                            ControllerContext c = this.ControllerContext;
                            c.HttpContext.Response.Redirect(c.HttpContext.Request.Url.ToString()+"?NumberHpoAccessible=0");
                        }
                    }
                }

                if (this.SessionConfiguration.RepresentingHospital != null)
                {
                    m.Hpo = this.GetHpoDisplayName(ObjectMapper.Map<HospitalViewModel>(this.SessionConfiguration.RepresentingHospital));
                    m.HpoLogoArray = this.GetHpoLogoArray(ObjectMapper.Map<HospitalViewModel>(this.SessionConfiguration.RepresentingHospital));
                    IEnumerable<HospitalViewModel> hospitals = SetHospitalForUser(); // see if this user has only 1 hospital assigned to them and default to that one
                    m.NumberHpoAccessible = (hospitals != null ? hospitals.Count() : 0);
                }
            }
            return this.PartialView(m);
        }
        /// <summary>
        /// Clears out the SessionConfiguration object and then calls the Account Logoff action
        /// </summary>
        /// <returns>Login View</returns>
        public ActionResult LogOff()
        {
            this.SessionConfiguration.RepresentingHospital = null; // Set to null
            this.hospitalRepository.Dispose();
            this.Session.Abandon();
            // call account logoff
            return RedirectToAction("LogOff", "Account");
        }

        #endregion Session Information

        #region Select Health Provider Organisation

        /// <summary>
        /// Support selection of a health provider organisation.
        /// </summary>
        /// <param name="m">View model.</param>
        /// <returns>View for selection of a health provider organisation.</returns>
        public ActionResult SelectHpo(SelectHpoViewModel m)
        {
            // Validate & default return URL if required.
            if (string.IsNullOrEmpty(m.ReturnUrl))
            {
                return this.RedirectToAction("SelectHpo", new { returnUrl = this.Url.Action("Index", "Home") });
            }

            if (this.TempData != null && this.TempData.ContainsKey("HpoRequiredMessage"))
            {
                m.Messages.Add((string)this.TempData["HpoRequiredMessage"], MessageLevel.Error);
            }
            // retrieve the Hospital lists the user has access to
            IEnumerable<HospitalViewModel> hospitals = SetHospitalForUser();

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                
                if (!string.IsNullOrEmpty(m.SelectedHospitalCode))
                {
                    var selectedHospital = hospitals.SingleOrDefault(h => h.Code == m.SelectedHospitalCode);
                    if (selectedHospital != null)
                    {
                        this.SessionConfiguration.RepresentingHospital = ObjectMapper.Map<Hospital>(selectedHospital);
                        return this.Redirect(m.ReturnUrl);
                    }
                    else
                    {
                        m.Messages.Add("The selected health provider organisation could not be found.", MessageLevel.Error);
                    }
                }
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => this.GetHpoDisplayName(h));
                if (this.SessionConfiguration.RepresentingHospital != null)
                {
                    m.SelectedHospitalCode = this.SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
                }
            }
            else
            {
                m.Messages.Add("No health provider organisations available for selection.", MessageLevel.Error);
            }
            return this.View(m);
        }
       
        #endregion Select Health Provider Organisation

        #endregion Actions

        #region Helpers

        /// <summary>
        /// Gets the display name for the specified hospital.
        /// </summary>
        /// <param name="hospital">Hospital view model.</param>
        /// <returns>String containing the display name.</returns>
        private string GetHpoDisplayName(HospitalViewModel hospital)
        {
            return (hospital.Name == hospital.HpioName) ? hospital.Name : string.Format("{0} ({1})", hospital.Name, hospital.HpioName);
        }
        /// <summary>
        /// Gets the array of bytes for the specified hospital logo.
        /// </summary>
        /// <param name="hospital">Hospital view model.</param>
        /// <returns>byte array or null.</returns>
        private byte[] GetHpoLogoArray(HospitalViewModel hospital)
        {
            return (hospital.HpioLogo == null ? null: hospital.HpioLogo);
        }
        /// <summary>
        /// Gets and sets the hospitals the user has access to based on their security groups/roles
        /// </summary>
        /// <returns>A enumerable collection of HospitalViewModels</returns>
        private IEnumerable<HospitalViewModel> SetHospitalForUser()
        {
            // Attempt to get identity from controller.
            ClaimsIdentity cid = null;
            if ((this.User != null) && (this.User.Identity != null))
            {
                // Claims identity created at login
                cid = (ClaimsIdentity)this.User.Identity;
            }
            // Get list of health provider organisations available for selection.
            IEnumerable<HospitalViewModel> accessibleHospitals = null;
            if (cid != null)
            {
                List<string> userSecurityGroups = new List<string>();
                foreach (var role in cid.FindAll(ClaimTypes.Role))
                {
                    var domain = role.Value.Contains("\\") ? role.Value.Split('\\')[0] : "";
                    userSecurityGroups.Add(role.Value.Replace(domain + "\\", string.Empty));
                }
                // Match security groups to HospitalCodes
                if (userSecurityGroups.Count > 0)
                {
                    accessibleHospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetAccessibleHospitals(userSecurityGroups, this.Settings.GetSettingValue(Setting.SettingCodes.SecurityGroupCodeSystem)), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));
                }
            }

            return accessibleHospitals;
        }
        
        #endregion Helpers

        #endregion Methods
    }
}