﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Web.Mvc;

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.ServiceModel;
using HIPS.Web.Components.Web;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.DocumentManagement;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.DocumentManagement;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for the document management feature.
    /// </summary>
    [HpoRequired]
    public class DocumentManagementController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Patient repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientRepository;

        /// <summary>
        /// Uploaded document repository to be used by this controller.
        /// </summary>
        private readonly IUploadedDocumentRepository uploadedDocumentRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="DocumentManagementController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="uploadedDocumentRepository">Uploaded document repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public DocumentManagementController(IHospitalRepository hospitalRepository, IPatientRepository patientRepository, ISettingsRepository settingsRepository, IUploadedDocumentRepository uploadedDocumentRepository, ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.patientRepository = patientRepository;
            this.uploadedDocumentRepository = uploadedDocumentRepository;
        }

        #endregion Constructors

        #region Methods

        #region Patient List Actions

        /// <summary>
        /// Display a list of patients for a selected hospital.
        /// </summary>
        /// <param name="lookupMessages">Additional messages to be displayed coming from the lookup MRN action.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Patients(ViewMessageList lookupMessages = null)
        {
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            // Create ViewModel:
            var m = new PatientsViewModel() { HospitalId = hospitalId };
            this.LoadCurrentContext(m);

            // Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                // Update ViewModel with reference data:
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                m.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            // Only load patients after a hospital has been selected.
            if (hospitalId != null)
            {
                // Load patients for selected hospital.
                var response = this.patientRepository.ListAdmittedPatients(
                    this.GetCurrentUserDetails(), this.DefaultHospitalCodeSystem, hospitalId, withIhi: null, withPcehr: null, excludeMedicareExclusions: false);

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    if (response.Data.AdmittedPatientList.Count() > 0)
                    {
                        // Update ViewModel with patients.
                        m.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(response.Data.AdmittedPatientList));
                    }
                    else
                    {
                        m.Messages.Add("There are no patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";
                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    m.Messages.Add(errorMessage, MessageLevel.Error);
                }
                if (lookupMessages != null)
                {
                    m.Messages.AddRange(lookupMessages);
                }
            }
            return this.View("Patients", m);
        }

        #endregion Patient List Actions

        #region Document List Actions

        /// <summary>
        /// Display a list of uploaded documents for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult Documents(string hospitalId, string patientId)
        {
            var m = new UploadedDocumentsViewModel { HospitalId = hospitalId, PatientId = patientId };
            this.LoadCurrentContext(m);
            Mrn patientIdentifier = new Mrn(m.PatientId, m.HospitalId, this.DefaultHospitalCodeSystem);
            var response = this.uploadedDocumentRepository.List(this.GetCurrentUserDetails(), null, patientIdentifier);
            if (response.Data.HipsResponse.Status == HipsResponseIndicator.OK || response.Data.HipsResponse.Status == HipsResponseIndicator.UnresolvedIhiAlert)
            {
                var source = response.Data.DocumentList;
                m.UploadedDocuments = ObjectMapper.Map<List<UploadedDocumentViewModel>>(source);
                if (source.Count == 0)
                {
                    m.Messages.Add("There are no documents for the selected patient.", MessageLevel.Information);
                }

                if (response.Data.HipsResponse.Status == HipsResponseIndicator.UnresolvedIhiAlert)
                {
                    m.IsIhiValid = false;
                    m.Messages.Add("Due to the unresolved IHI alert, the documents could be viewed but not removed", MessageLevel.Warning);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve documents for the selected patient.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
                m.Messages.Add(errorMessage, MessageLevel.Error);
            }
            return this.View("Documents", m);
        }

        /// <summary>
        /// Look up a patient by MRN. If found then display the list of
        /// documents for that patient, otherwise redisplay the list of patients
        /// with a message indicating that the patient was not found.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="lookupMrn">User-entered identifier of the patient.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult LookupMrnForRemoveDocument(string hospitalId, string lookupMrn)
        {
            var m = new UploadedDocumentsViewModel { HospitalId = hospitalId, PatientId = lookupMrn };
            this.LoadCurrentContext(m);

            if (m.CurrentPatient != null)
            {
                return this.Documents(hospitalId, lookupMrn);
            }
            else
            {
                ViewMessageList messages = new ViewMessageList();
                messages.Add("There is no current patient with this MRN.", MessageLevel.Error);
                return this.Patients(messages);
            }
        }

        #endregion Document List Actions

        #region View Document Actions

        /// <summary>
        /// Display the selected document. If invoked via AJAX, the result is a
        /// partial view to be loaded into a modal popup, otherwise it is a
        /// complete page. The view shows a progress bar while it loads the
        /// "DocumentContent" action.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="setId">Identifier of the document.</param>
        /// <returns>The view or partial view result.</returns>
        [HttpGet]
        public ActionResult ViewDocument(string hospitalId, string patientId, string setId)
        {
            // Create ViewModel:
            DocumentManagementViewModelBase m = new DocumentManagementViewModelBase() { HospitalId = hospitalId, PatientId = patientId, SetId = setId };
            this.LoadCurrentContext(m);

            // If Ajax request return a PartialView
            if (this.Request != null && this.Request.IsAjaxRequest())
            {
                return this.PartialView(m);
            }

            return this.View(m);
        }

        /// <summary>
        /// Display the selected document. The result is a partial view to
        /// be loaded into the container on the ViewDocument page. The view
        /// contains an informational message regarding the source of the
        /// document, and an i-frame in which the rendered HTML will be shown.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="setId">Identifier of the document.</param>
        /// <returns>The partial view result.</returns>
        [HttpGet]
        public ActionResult DocumentContent(string hospitalId, string patientId, string setId)
        {
            DocumentContentViewModel m = new DocumentContentViewModel() { HospitalId = hospitalId, PatientId = patientId, SetId = setId };
            this.LoadCurrentContext(m);

            // Retrieve document
            var doc = this.uploadedDocumentRepository.Get(this.GetCurrentUserDetails(), setId, null, new Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem));

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(doc.Messages));

            // Save and transform if a document was retrieved
            if (doc.IsSuccessful)
            {
                m.DocumentUrl = Url.Action("DocumentContentView", new { hospitalId, patientId, setId, filename = "CDA_ROOT.html" });
                m.UploadTimestamp = doc.Data.LocalClinicalDocumentMetaData.UploadedDate;
            }
            else
            {
                string errorMessage = "Failed to retrieve content for the selected document.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, doc.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return this.PartialView(m);
        }

        /// <summary>
        /// Gets a file from a document package. For the file name "CDA_ROOT.html"
        /// it will return the HTML that is produced by applying the NEHTA
        /// style sheet to the CDA document. For the file name "CDA_ROOT.xml" it
        /// will return the raw XML source, which is not required for the web
        /// site but is useful for testing. For other file names, it will return
        /// the attachment file.
        /// </summary>
        /// <param name="hospitalId">Identifier of the selected hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="setId">Identifier of the selected document.</param>
        /// <param name="filename">Name of the file that is requested.</param>
        /// <returns>HTML Content, File or FileInline.</returns>
        [HttpGet]
        public ActionResult DocumentContentView(string hospitalId, string patientId, string setId, string filename)
        {
            // Check if request is for a filename that the filename is valid
            if (!string.IsNullOrWhiteSpace(filename))
            {
                // If filename is missing an extension, or has a directory, then it is invalid
                bool hasDirectory = !string.IsNullOrWhiteSpace(Path.GetDirectoryName(filename));
                if (!Path.HasExtension(filename) || hasDirectory)
                {
                    return this.HttpNotFound();
                }
            }

            // Retrieve document
            var doc = this.uploadedDocumentRepository.Get(this.GetCurrentUserDetails(), setId, null, new Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem));

            // If unsuccessful lookup then return 404
            if (!doc.IsSuccessful)
            {
                return this.HttpNotFound();
            }

            // Looking for rendered HTML (no filename or filename is "CDA_ROOT.html")
            if (filename == null || filename.Equals("CDA_ROOT.html", StringComparison.OrdinalIgnoreCase))
            {
                return this.Content(doc.Data.TransformToHtml(Properties.Resources.NEHTA_Generic_CDA_Stylesheet_1_2_8));
            }

            // Looking for XML source
            if (filename.Equals("CDA_ROOT.xml", StringComparison.OrdinalIgnoreCase))
            {
                return this.File(doc.Data.Document, "text/xml");
            }

            // Looking for an attachment
            {
                var attachment = doc.Data.Attachments.FirstOrDefault(a => a.FileName.Equals(filename, StringComparison.OrdinalIgnoreCase));

                // If nothing found, return 404
                if (attachment == null)
                {
                    return this.HttpNotFound();
                }

                // Return attachment with filename and inferred MIME type, attempt to display inline rather than prompt download
                return this.FileInlineInferMime(attachment.Contents, filename);
            }
        }

        #endregion View Document Actions

        #region Remove Document Actions

        /// <summary>
        /// Display a form for the user to administer the removal of a selected document.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the patient.</param>
        /// <param name="setId">Identifier of the document set.</param>
        /// <returns>The view result.</returns>
        [HttpGet]
        public ActionResult RemoveDocument(string hospitalId, string patientId, string setId)
        {
            var m = new RemoveDocumentViewModel { HospitalId = hospitalId, PatientId = patientId, SetId = setId };
            this.LoadCurrentContext(m);

            // Set an invalid value to ensure that neither radio button is checked by default.
            m.Reason = (RemovalReason)(-1);

            // If Ajax request return a PartialView
            if (this.Request != null && this.Request.IsAjaxRequest())
            {
                return this.PartialView(m);
            }
            return this.View(m);
        }

        /// <summary>
        /// Process the submission of the form to remove a document.
        /// </summary>
        /// <param name="model">Submitted form model.</param>
        /// <returns>JSON response for display on screen.</returns>
        [HttpPost]
        public ActionResult RemoveDocument(RemoveDocumentViewModel model)
        {
            var response = new HIPS.Web.Components.ServiceModel.ServiceResponse<HipsResponse>(null, false, new ResponseMessageList());
            if (ModelState.IsValid)
            {
                this.LoadCurrentContext(model);
                bool stubOut = false;
                if (stubOut)
                {
                    response.Messages.Add("Currently stubbed out.", MessageLevel.Information);
                }
                else
                {
                    Mrn patientIdentifier = new Mrn(model.PatientId, model.HospitalId, model.CurrentHospital.CodeSystemCode);
                    response = this.uploadedDocumentRepository.Remove(
                        patientIdentifier,
                        model.CurrentDocument.AdmissionDate,
                        model.CurrentDocument.SourceSystemSetId,
                        model.Reason,
                        this.GetCurrentUserDetails(),
                        Encoding.UTF8.GetBytes(model.Notes));
                }
            }
            return this.Json(new { Response = response, Echo = model, Errors = ModelState.ToErrorDictionary() });
        }

        #endregion Remove Document Actions

        #region Helpers

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        private void LoadCurrentContext(DocumentManagementViewModelBase model)
        {
            var mrn = new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem);

            // Load current hospital.
            if (!string.IsNullOrEmpty(model.HospitalId))
            {
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);
                }
            }

            // Load current patient.
            if (!string.IsNullOrEmpty(model.PatientId))
            {
                var response = this.patientRepository.GetPatientDisclosureDetails(this.GetCurrentUserDetails(), mrn);
                if (response.IsSuccessful)
                {
                    model.CurrentPatient = ObjectMapper.Map<PatientViewModel>(response.Data.AdmittedPatient);
                }
            }

            // Load current document
            if (!string.IsNullOrEmpty(model.SetId))
            {
                var documentResponse = this.uploadedDocumentRepository.Get(this.GetCurrentUserDetails(), model.SetId, null, mrn);

                if (documentResponse.IsSuccessful)
                {
                    var document = documentResponse.Data.LocalClinicalDocumentMetaData;
                    model.CurrentDocument = ObjectMapper.Map<UploadedDocumentViewModel>(document);
                }
                else
                {
                    throw new System.Exception("Could not retrieve specified document.");
                }
            }
        }

        #endregion Helpers

        #endregion Methods
    }
}