using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using System.Web.Mvc;

using HIPS.CommonSchemas.Hi;
using HIPS.HpiiSchemas;
using HIPS.Web.Components.Cache;
using HIPS.Web.Components.Collections;
using HIPS.Web.Data.Hips.HpiiSearch;
using HIPS.Web.Data.Hips.Reference;
using HIPS.Web.Data.WebsiteDb;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.Hi;
using HIPS.Web.ModelInterface.HpiiSearch;
using HIPS.Web.UI.Conversion.HpiiSearch;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.ViewModels.HpiiSearch;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for HPI-I search.
    /// </summary>
    [HpoRequired]
    public class HpiiSearchController : ControllerBase
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="HpiiSearchController" /> class.
        /// </summary>
        public HpiiSearchController()
            : this(
                new HpiiSearchService(),
                new CachedHiReferenceRepository(new HiReferenceRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))),
                new CachedSettingsRepository(new WebsiteDbRepository(), new WebMemoryCacheProvider(TimeSpan.FromHours(2))),
                new SessionConfiguration(new HttpSessionProvider()))
        {
        }

        /// <summary>
        /// Initialises a new instance of the <see cref="HpiiSearchController" /> class.
        /// </summary>
        /// <param name="hpiiSearchService">The HPI-I search service to be used by this controller.</param>
        /// <param name="healthcareIdentifierReferenceRepository">The HI reference repository to be used by this controller.</param>
        /// <param name="settingsRepository">The settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public HpiiSearchController(IHpiiSearchService hpiiSearchService, IHiReferenceRepository healthcareIdentifierReferenceRepository, ISettingsRepository settingsRepository, ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.HpiiSearchService = hpiiSearchService;
            this.HiReferenceRepository = healthcareIdentifierReferenceRepository;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the HPI-I search service to be used by this controller.
        /// </summary>
        private IHpiiSearchService HpiiSearchService { get; set; }

        /// <summary>
        /// Gets or sets the HI reference repository to be used by this controller.
        /// </summary>
        private IHiReferenceRepository HiReferenceRepository { get; set; }

        /// <summary>
        /// Gets or sets the hospital repository to be used by this controller.
        /// </summary>
        private IHospitalRepository HospitalRepository { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Displays the HPI-I identifier search form.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult SearchById()
        {
            // Create ViewModel
            SearchByIdViewModel viewModel = new SearchByIdViewModel();

            // Load reference data
            this.LoadAndAddReferenceData(viewModel);

            // Render view
            return this.View(viewModel);
        }

        /// <summary>
        /// Processes the submission of the HPI-I identifier search form.
        /// </summary>
        /// <param name="searchById">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult SearchById(SearchByIdViewModel searchById)
        {
            searchById.Hpio = this.SessionConfiguration.RepresentingHospital.Hpio;

            // Load reference data
            this.LoadAndAddReferenceData(searchById);

            // Check basic validation errors
            if (!this.ModelState.IsValid)
            {
                // Return view (with original ViewModel)
                return this.View(searchById);
            }

            // Invoke service
            HpiiQueryResponse queryResponse;
            try
            {
                queryResponse = this.HpiiSearchService.SearchByIdentifier(searchById.ToIdentifierQuery(), this.GetCurrentUserDetails());
            }
            catch (EndpointNotFoundException endPoint)
            {
                // Add Model State error
                this.ModelState.AddModelError(string.Empty, "Unable to Connect to HIPS so the HPI-I cannot be validated.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(endPoint);
                // Render view (with original ViewModel)
                return this.View(searchById);
            }
            catch (Exception e)
            {
                // Add Model State error (TODO: Remove this when HIPS provides improved validation support)
                this.ModelState.AddModelError(string.Empty, "Unable to locate HPI-I.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(e);
                // Render view (with original ViewModel)
                return this.View(searchById);
            }

            // Convert to response ViewModel
            SearchResultViewModel responseVm = new SearchResultViewModel().LoadFromResponse(queryResponse);

            // Set View Logic Properties
            responseVm.ShowAustralianAddress = false;
            responseVm.ShowInternationalAddress = false;

            // Load reference data
            this.LoadAndAddReferenceData(responseVm, responseVm.ShowAustralianAddress, responseVm.ShowInternationalAddress);

            // If service errors, or "Not Found", return
            if (responseVm.ResponseMessages.Errors.Any() || queryResponse.HipsResponse.ResponseCode == "WSE0035")
            {
                // Set the Response Messages on the View Model being returned
                searchById.ResponseMessages = responseVm.ResponseMessages;

                // Render view (with original ViewModel)
                return this.View(searchById);
            }

            // Add success to view model
            responseVm.ResponseMessages.Successes.Add("HPI-I Search result found.");

            // Render response success view
            return this.View("SearchResult", responseVm);
        }

        /// <summary>
        /// Displays the HPI-I demographic search form.
        /// </summary>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult SearchByDemographics()
        {
            // Create ViewModel
            SearchByDemographicsViewModel viewModel = new SearchByDemographicsViewModel();

            // Load reference data
            this.LoadAndAddReferenceData(viewModel);

            // Render view
            return this.View(viewModel);
        }

        /// <summary>
        /// Processes the submission of the HPI-I demographic search form.
        /// </summary>
        /// <param name="searchByDemographics">View model.</param>
        /// <returns>View result.</returns>
        [HttpPost]
        public ActionResult SearchByDemographics(SearchByDemographicsViewModel searchByDemographics)
        {
            searchByDemographics.Hpio = this.SessionConfiguration.RepresentingHospital.Hpio;

            // Load reference data
            this.LoadAndAddReferenceData(searchByDemographics);

            // Check basic validation errors
            if (!this.ModelState.IsValid)
            {
                // Return view (with original ViewModel)
                return this.View(searchByDemographics);
            }

            // Invoke service
            HpiiQueryResponse queryResponse;
            try
            {
                queryResponse = this.HpiiSearchService.SearchByDemographic(searchByDemographics.ToDemographicQuery(), this.GetCurrentUserDetails());
            }
            catch (EndpointNotFoundException endPoint)
            {
                // Add Model State error
                this.ModelState.AddModelError(string.Empty, "Unable to Connect to HIPS so the HPI-I cannot be validated.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(endPoint);
                // Render view (with original ViewModel)
                return this.View(searchByDemographics);
            }
            catch (Exception e)
            {
                // Add Model State error (TODO: Remove this when HIPS provides improved validation support)
                this.ModelState.AddModelError(string.Empty, "Unable to locate HPI-I.");
                Elmah.ErrorSignal.FromCurrentContext().Raise(e);
                // Render view (with original ViewModel)
                return this.View(searchByDemographics);
            }

            // Convert to response ViewModel
            SearchResultViewModel responseVm = new SearchResultViewModel().LoadFromResponse(queryResponse);

            // Set View Logic Properties
            responseVm.ShowInternationalAddress = responseVm.InternationalAddress.HasAnyValue();
            responseVm.ShowAustralianAddress = !responseVm.ShowInternationalAddress;

            // Load reference data
            this.LoadAndAddReferenceData(responseVm, responseVm.ShowAustralianAddress, responseVm.ShowInternationalAddress);

            // If service errors, or "Not Found", return
            if (responseVm.ResponseMessages.Errors.Any() || queryResponse.HipsResponse.ResponseCode == "WSE0035")
            {
                // Set the Response Messages on the View Model being returned
                searchByDemographics.ResponseMessages = responseVm.ResponseMessages;

                // Render view (with original ViewModel)
                return this.View(searchByDemographics);
            }

            // Add success to view model
            responseVm.ResponseMessages.Successes.Add("HPI-I Search result found.");

            // Render response success view
            return this.View("SearchResult", responseVm);
        }

        /// <summary>
        /// Releases unmanaged resources and optionally releases managed resources.
        /// </summary>
        /// <param name="disposing">true to release both managed and unmanaged resources; false to release only unmanaged resources.</param>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                if (this.HpiiSearchService != null)
                {
                    this.HpiiSearchService.Dispose();
                    this.HpiiSearchService = null;
                }

                if (this.HiReferenceRepository != null)
                {
                    this.HiReferenceRepository.Dispose();
                    this.HiReferenceRepository = null;
                }

                if (this.HospitalRepository != null)
                {
                    this.HospitalRepository.Dispose();
                    this.HospitalRepository = null;
                }
            }

            base.Dispose(disposing);
        }

        #region Helper Methods

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="searchByIdViewModel">View model.</param>
        /// <returns>The given view model, for fluency.</returns>
        private SearchByIdViewModel LoadAndAddReferenceData(SearchByIdViewModel searchByIdViewModel)
        {
            // Load reference data
            List<HiSex> sexes = this.HiReferenceRepository.GetHiSexes();
            List<HiState> states = this.HiReferenceRepository.GetHiStates();

            // Update ViewModel with reference data
            searchByIdViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByIdViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);

            // Return for fluency
            return searchByIdViewModel;
        }

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="searchByDemographicsViewModel">View model.</param>
        /// <returns>The given view model, for fluency.</returns>
        private SearchByDemographicsViewModel LoadAndAddReferenceData(SearchByDemographicsViewModel searchByDemographicsViewModel)
        {
            // Load reference data
            List<HiSex> sexes = this.HiReferenceRepository.GetHiSexes();
            List<HiState> states = this.HiReferenceRepository.GetHiStates();
            List<HiUnitType> unitTypes = this.HiReferenceRepository.GetHiUnitTypes();
            List<HiLevelType> levelTypes = this.HiReferenceRepository.GetHiLevelTypes();
            List<HiStreetType> streetTypes = this.HiReferenceRepository.GetHiStreetTypes();
            List<HiStreetSuffixType> streetSuffixes = this.HiReferenceRepository.GetHiStreetSuffixTypes();
            List<HiPostalDeliveryType> postalDeliveryTypes = this.HiReferenceRepository.GetHiPostalDeliveryTypes();
            List<HiCountry> countries = this.HiReferenceRepository.GetHiCountries();

            // Update ViewModel with reference data
            searchByDemographicsViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.UnitTypes = unitTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.LevelTypes = levelTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.StreetTypes = streetTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.StreetSuffixes = streetSuffixes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.PostalDeliveryTypes = postalDeliveryTypes.ToSelectListItems(s => s.Code, s => s.Description);
            searchByDemographicsViewModel.Countries = countries.ToSelectListItems(s => s.Code, s => s.Description);

            return searchByDemographicsViewModel;
        }

        /// <summary>
        /// Loads reference data into the view model.
        /// </summary>
        /// <param name="searchResultViewModel">View model.</param>
        /// <param name="usingAustralianAddress">Whether australian addresses are used.</param>
        /// <param name="usingInternationalAddress">Whether international addresses are used.</param>
        /// <returns>The given view model, for fluency.</returns>
        private SearchResultViewModel LoadAndAddReferenceData(SearchResultViewModel searchResultViewModel, bool usingAustralianAddress, bool usingInternationalAddress)
        {
            // Load reference data
            List<HiSex> sexes = this.HiReferenceRepository.GetHiSexes();
            List<HiState> states = this.HiReferenceRepository.GetHiStates();

            // Update ViewModel with reference data
            searchResultViewModel.Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
            searchResultViewModel.States = states.ToSelectListItems(s => s.Code, s => s.Description);

            if (usingAustralianAddress)
            {
                // Load reference data
                List<HiUnitType> unitTypes = this.HiReferenceRepository.GetHiUnitTypes();
                List<HiLevelType> levelTypes = this.HiReferenceRepository.GetHiLevelTypes();
                List<HiStreetType> streetTypes = this.HiReferenceRepository.GetHiStreetTypes();
                List<HiStreetSuffixType> streetSuffixes = this.HiReferenceRepository.GetHiStreetSuffixTypes();
                List<HiPostalDeliveryType> postalDeliveryTypes = this.HiReferenceRepository.GetHiPostalDeliveryTypes();

                // Update ViewModel with reference data
                searchResultViewModel.UnitTypes = unitTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.LevelTypes = levelTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.StreetTypes = streetTypes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.StreetSuffixes = streetSuffixes.ToSelectListItems(s => s.Code, s => s.Description);
                searchResultViewModel.PostalDeliveryTypes = postalDeliveryTypes.ToSelectListItems(s => s.Code, s => s.Description);
            }
            if (usingInternationalAddress)
            {
                // Load reference data
                List<HiCountry> countries = this.HiReferenceRepository.GetHiCountries();

                // Update ViewModel with reference data
                searchResultViewModel.Countries = countries.ToSelectListItems(s => s.Code, s => s.Description);
            }

            // Return for fluency
            return searchResultViewModel;
        }

        #endregion Helper Methods

        #endregion Methods
    }
}