using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Web.Mvc;

using HIPS.PcehrSchemas;
using HIPS.Web.Components.Collections;
using HIPS.Web.Components.Common;
using HIPS.Web.Components.Web;
using HIPS.Web.Model.Common;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.ModelInterface.PcehrView;
using HIPS.Web.UI.Filters;
using HIPS.Web.UI.Helpers;
using HIPS.Web.UI.Helpers.Mapping;
using HIPS.Web.UI.ViewModels.PcehrView;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.Controllers
{
    /// <summary>
    /// Controller for the "PCEHR Web Viewer" feature.
    /// </summary>
    [NoCache]
    [HpoRequired]
    public class PcehrViewController : ControllerBase
    {
        #region Fields

        /// <summary>
        /// Gets the hospital repository to be used by this controller.
        /// </summary>
        private readonly IHospitalRepository hospitalRepository;

        /// <summary>
        /// Gets the patient repository to be used by this controller.
        /// </summary>
        private readonly IPatientRepository patientRepository;

        /// <summary>
        /// Gets the PCEHR view repository to be used by this controller.
        /// </summary>
        private readonly IPcehrViewRepository pcehrViewRepository;

        #endregion Fields

        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="PcehrViewController" /> class.
        /// </summary>
        /// <param name="hospitalRepository">Hospital repository to be used by this controller.</param>
        /// <param name="patientRepository">Patient repository to be used by this controller.</param>
        /// <param name="pcehrViewRepository">PCEHR view repository to be used by this controller.</param>
        /// <param name="settingsRepository">Settings repository to be used by this controller.</param>
        /// <param name="sessionConfiguration">Session configuration to be used by this controller.</param>
        public PcehrViewController(IHospitalRepository hospitalRepository, IPatientRepository patientRepository, IPcehrViewRepository pcehrViewRepository, ISettingsRepository settingsRepository, ISessionConfiguration sessionConfiguration)
            : base(settingsRepository, sessionConfiguration)
        {
            this.hospitalRepository = hospitalRepository;
            this.patientRepository = patientRepository;
            this.pcehrViewRepository = pcehrViewRepository;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets a list containing prescription and dispense document classes.
        /// </summary>
        private List<string> PrescriptionAndDispenseDocumentClasses
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.PcehrViewPDDocumentClasses).Split(',').ToList();
            }
        }

        /// <summary>
        /// Gets a list containing document classes for which service dates should be displayed.
        /// </summary>
        private List<string> ServiceDateDocumentClasses
        {
            get
            {
                return this.Settings.GetSettingValue(Setting.SettingCodes.PcehrViewServiceDateDocumentClasses).Split(',').ToList();
            }
        }

        /// <summary>
        /// Gets an integer value representing the number of months the "From" date will be offset from the current date by default.
        /// </summary>
        private int FromDateOffsetMonths
        {
            get
            {
                return int.Parse(this.Settings.GetSettingValue(Setting.SettingCodes.PcehrViewFromDateOffsetMonths));
            }
        }

        #endregion Properties

        #region Methods

        #region Actions

        #region Patients at Hospital

        /// <summary>
        /// Display a list of patients for a selected hospital.
        /// </summary>
        /// <returns>View result</returns>
        [HttpGet]
        public ActionResult Patients()
        {
            string hospitalId = SessionConfiguration.RepresentingHospital.HospitalFacilityCode;
            // Create ViewModel:
            var m = new PatientsViewModel() { HospitalId = hospitalId };
            this.LoadCurrentContext(m);

            // Load reference data:
            var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

            if ((hospitals != null) && (hospitals.Count() > 0))
            {
                // Update ViewModel with reference data:
                m.Hospitals = hospitals.ToSelectListItems(h => h.Code, h => h.Name);
            }
            else
            {
                m.Messages.Add("No hospitals available for selection.", MessageLevel.Error);
            }

            // Don't allow load for all patients:
            if (hospitalId != null)
            {
                // Load patients for selected hospital.
                var response = this.patientRepository.ListPatientsCurrentlyInHospital(this.DefaultHospitalCodeSystem, hospitalId, true, this.GetCurrentUserDetails());

                // Ensure loading was successful.
                if (response.IsSuccessful)
                {
                    if (response.Data.PatientInHospitalList.Count > 0)
                    {
                        // Update ViewModel with patients.
                        m.Patients.AddRange(ObjectMapper.Map<IEnumerable<PatientViewModel>>(response.Data.PatientInHospitalList));
                    }
                    else
                    {
                        m.Messages.Add("No PCEHR data is available for patients at the selected hospital.", MessageLevel.Information);
                    }
                }
                else
                {
                    string errorMessage = "Failed to retrieve patients for the selected hospital.";
                    // Log details:
                    Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, response.Messages.AsString())));
                    // Display error message.
                    this.SetAjaxErrorResponseCode();
                    m.Messages.Add(errorMessage, MessageLevel.Error);
                }
            }

            return this.View("Patients", m);
        }

        #endregion Patients at Hospital

        #region Patient Summary

        /// <summary>
        /// Display a summary for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        [PcehrAccess]
        public ActionResult PatientSummary(string hospitalId, string patientId)
        {
            // Create ViewModel:
            var m = new PatientSummaryViewModel() { HospitalId = hospitalId, PatientId = patientId };
            this.LoadCurrentContext(m);

            // Load documents for selected patient.
            var docs = this.pcehrViewRepository.ListActiveDocuments(
                new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails());

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(docs.Messages));

            // Load document list
            if (docs.IsSuccessful)
            {
                if (docs.Data.DocumentList != null && docs.Data.DocumentList.Any())
                {
                    var includedDocuments = docs.Data.DocumentList.Where(d => !this.PrescriptionAndDispenseDocumentClasses.Contains(d.DocumentClassCode));
                    // Update ViewModel with documents & document categories.
                    m.Documents.AddRange(ObjectMapper.Map<IEnumerable<DocumentViewModel>>(includedDocuments));
                    m.DocumentCategories.AddRange(
                        includedDocuments
                            .GroupBy(d => new { d.DocumentClassCode, d.DocumentClassName })
                            .Select(cat => new DocumentCategoryViewModel()
                                {
                                    CategoryCode = cat.Key.DocumentClassCode,
                                    CategoryName = cat.Key.DocumentClassName,
                                    Count = cat.Count(),
                                    AreServiceDatesRelevant = this.ServiceDateDocumentClasses.Contains(cat.Key.DocumentClassCode)
                                })
                            .OrderBy(cat => cat.CategoryName));
                    m.PrescriptionDispenseDocumentCount = docs.Data.DocumentList.Count(d => this.PrescriptionAndDispenseDocumentClasses.Contains(d.DocumentClassCode));
                }
                else
                {
                    m.Messages.Add("No documents are available for the selected patient.", MessageLevel.Information);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve documents for the selected patient.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, docs.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return this.View(m);
        }

        #endregion Patient Summary

        #region Enterprise Patient

        /// <summary>
        /// Display a summary for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        [PcehrAccess]
        public ActionResult EnterprisePatientSummary(string hospitalId, string patientId, string statePatientId)
        {
            // Create ViewModel:
            var m = new PatientSummaryViewModel() { HospitalId = hospitalId, PatientId = patientId };
            this.LoadCurrentContext(m);

            // Load documents for selected patient.
            var docs = this.pcehrViewRepository.ListActiveDocuments(
                new CommonSchemas.PatientIdentifier.RegisteredEnterprisePatient(patientId, string.IsNullOrEmpty(statePatientId) ? patientId : statePatientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails());

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(docs.Messages));

            // Load document list
            if (docs.IsSuccessful)
            {
                if (docs.Data.DocumentList != null && docs.Data.DocumentList.Any())
                {
                    var includedDocuments = docs.Data.DocumentList.Where(d => !this.PrescriptionAndDispenseDocumentClasses.Contains(d.DocumentClassCode));
                    // Update ViewModel with documents & document categories.
                    m.Documents.AddRange(ObjectMapper.Map<IEnumerable<DocumentViewModel>>(includedDocuments));
                    m.DocumentCategories.AddRange(
                        includedDocuments
                            .GroupBy(d => new { d.DocumentClassCode, d.DocumentClassName })
                            .Select(cat => new DocumentCategoryViewModel()
                            {
                                CategoryCode = cat.Key.DocumentClassCode,
                                CategoryName = cat.Key.DocumentClassName,
                                Count = cat.Count(),
                                AreServiceDatesRelevant = this.ServiceDateDocumentClasses.Contains(cat.Key.DocumentClassCode)
                            })
                            .OrderBy(cat => cat.CategoryName));
                    m.PrescriptionDispenseDocumentCount = docs.Data.DocumentList.Count(d => this.PrescriptionAndDispenseDocumentClasses.Contains(d.DocumentClassCode));
                }
                else
                {
                    m.Messages.Add("No documents are available for the selected patient.", MessageLevel.Information);
                }
            }
            else
            {
                string errorMessage = "Failed to retrieve documents for the selected patient.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, docs.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return this.View(m);
        }

        #endregion Enterprise Patient

        #region Document / Content

        /// <summary>
        /// Display a specific document for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="repositoryId">Identifier of the document repository.</param>
        /// <param name="documentId">Identifier of the document in the repository.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        [PcehrAccess]
        public ActionResult Document(string hospitalId, string patientId, string repositoryId, string documentId)
        {
            // Create ViewModel:
            PcehrViewModelBase m = new PcehrViewModelBase() { HospitalId = hospitalId, PatientId = patientId, RepositoryId = repositoryId, DocumentId = documentId };
            this.LoadCurrentContext(m);

            // If Ajax request return a PartialView
            if (Request.IsAjaxRequest())
            {
                return this.PartialView(m);
            }

            return this.View(m);
        }

        /// <summary>
        /// Gets a file from a downloaded document package. For the file name
        /// "CDA_ROOT.html" it will return the HTML that is produced by
        /// applying the NEHTA style sheet to the CDA document. For the file
        /// name "CDA_ROOT.xml" it will return the raw XML source, which is not
        /// required for the web site but is useful for testing. For other file
        /// names, it will return the attachment file.
        /// </summary>
        /// <param name="hospitalId">Identifier of the selected hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="repositoryId">Identifier of the PCEHR repository that holds the document.</param>
        /// <param name="documentId">Identifier of the document in the repository.</param>
        /// <param name="filename">Name of the file that is requested.</param>
        /// <returns>HTML Content, File or FileInline.</returns>
        [HttpGet]
        public ActionResult DocumentContentView(string hospitalId, string patientId, string repositoryId, string documentId, string filename)
        {
            // Check if request is for a filename that the filename is valid
            if (!string.IsNullOrWhiteSpace(filename))
            {
                // If filename is missing an extension, or has a directory, then it is invalid
                bool hasDirectory = !string.IsNullOrWhiteSpace(Path.GetDirectoryName(filename));
                if (!Path.HasExtension(filename) || hasDirectory)
                {
                    return this.HttpNotFound();
                }
            }

            // Retrieve document
            var doc = this.pcehrViewRepository.GetDocument(
                new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(),
                new DocumentRequest()
                {
                    RepositoryUniqueId = repositoryId,
                    DocumentUniqueId = documentId,
                    SaveDocument = false
                });

            // If unsuccessful lookup then return 404
            if (!doc.IsSuccessful)
            {
                return this.HttpNotFound();
            }

            // Looking for rendered HTML (no filename or filename is "CDA_ROOT.html")
            if (filename == null || filename.Equals("CDA_ROOT.html", StringComparison.OrdinalIgnoreCase))
            {
                return this.Content(doc.Data.TransformToHtml(Properties.Resources.NEHTA_Generic_CDA_Stylesheet_1_2_8));
            }

            // Looking for XML source
            if (filename.Equals("CDA_ROOT.xml", StringComparison.OrdinalIgnoreCase))
            {
                return this.File(doc.Data.Document, "text/xml");
            }

            // Looking for an attachment
            {
                var attachment = doc.Data.Attachments.FirstOrDefault(a => a.FileName.Equals(filename, StringComparison.OrdinalIgnoreCase));

                // If nothing found, return 404
                if (attachment == null)
                {
                    return this.HttpNotFound();
                }

                // Return attachment with filename and inferred MIME type, attempt to display inline rather than prompt download
                return this.FileInlineInferMime(attachment.Contents, filename);
            }
        }

        /// <summary>
        /// Display a specific document for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="repositoryId">Identifier of the document repository.</param>
        /// <param name="documentId">Identifier of the document in the repository.</param>
        /// <returns>View result.</returns>
        [HttpGet]
        public ActionResult DocumentContent(string hospitalId, string patientId, string repositoryId, string documentId)
        {
            DocumentContentViewModel m = new DocumentContentViewModel() { HospitalId = hospitalId, PatientId = patientId, RepositoryId = repositoryId, DocumentId = documentId };
            this.LoadCurrentContext(m);

            // Retrieve document
            var doc = this.pcehrViewRepository.GetDocument(
                new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(),
                new DocumentRequest()
                {
                    RepositoryUniqueId = repositoryId,
                    DocumentUniqueId = documentId,
                    SaveDocument = false
                });

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(doc.Messages));

            // Save and transform if a document was retrieved
            if (doc.IsSuccessful)
            {
                m.DocumentUrl = Url.Action("DocumentContentView", new { hospitalId, patientId, repositoryId, documentId, filename = "CDA_ROOT.html" });
                m.DownloadTimestamp = doc.ReceiptTimestamp;
            }
            else
            {
                string errorMessage = "Failed to retrieve content for the selected document.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, doc.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return this.PartialView(m);
        }

        #endregion Document / Content

        #region Prescription Dispense View / Content

        /// <summary>
        /// Display a prescription and dispense view page for a selected patient.
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns>View result</returns>
        [HttpGet]
        [PcehrAccess]
        public ActionResult PrescriptionDispenseView(PrescriptionDispenseViewModel request)
        {
            this.LoadCurrentContext(request);

            request.GroupByOptions = new Dictionary<string, string>()
            {
                { "prescription", "Prescription" },
                { "genericName", "Generic Name" },
                { "pbsItemCode", "PBS Item Code" },
                { "brandName", "Brand Name" },
            }.ToSelectListItems(i => i.Key, i => i.Value);

            if (!request.FromDate.HasValue || !request.ToDate.HasValue)
            {
                // Don't make the date fields required on page load of view
                ModelState.Clear();
            }

            request.FromDate = request.FromDate ?? DateTime.Today.AddMonths(this.FromDateOffsetMonths);
            request.ToDate = request.ToDate ?? DateTime.Today;
            request.GroupBy = request.GroupBy ?? request.GroupByOptions.First().Value;

            return this.PartialView(request);
        }

        /// <summary>
        /// Renders the content section of a prescription and dispense view page for a selected patient.
        /// </summary>
        /// <param name="request">The request.</param>
        /// <returns>View result</returns>
        public ActionResult PrescriptionDispenseViewContent(PrescriptionDispenseViewModel request)
        {
            // Create ViewModel (with breadcrumbs)
            DocumentContentViewModel m = new DocumentContentViewModel() { HospitalId = request.HospitalId, PatientId = request.PatientId };
            this.LoadCurrentContext(m);

            // If invalid, return with errors
            if (!ModelState.IsValid)
            {
                // Add ModelState errors to 'Messages'
                m.Messages.AddRange(ModelState.ToErrorDictionary().SelectMany(msg => msg.Value).Select(e => new ViewMessage(e, MessageLevel.Error)));
                this.SetAjaxErrorResponseCode();
                return this.PartialView("DocumentContent", m);
            }

            // Retrieve view
            var view = this.pcehrViewRepository.GetView(
                new CommonSchemas.PatientIdentifier.Mrn(request.PatientId, request.HospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(),
                new PrescriptionAndDispenseViewRequest() { FromDate = request.FromDate.Value, ToDate = request.ToDate.Value });

            // Copy over response messages.
            m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(view.Messages));

            // Return URL of view
            if (view.IsSuccessful)
            {
                m.DocumentUrl = Url.Action("PrescriptionDispenseViewContentView", new { FromDate = request.FromDate.Value.ToString("dd/MM/yyyy"), ToDate = request.ToDate.Value.ToString("dd/MM/yyyy"), GroupBy = request.GroupBy });
                m.DownloadTimestamp = view.ReceiptTimestamp;
            }
            else
            {
                string errorMessage = "Failed to retrieve content for the selected view.";
                // Log details:
                Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, view.Messages.AsString())));
                // Display error message.
                this.SetAjaxErrorResponseCode();
            }

            return this.PartialView("DocumentContent", m);
        }

        /// <summary>
        /// Gets the rendered HTML for the Prescription and Dispense View.
        /// </summary>
        /// <param name="request">View model containing the request.</param>
        /// <returns>HTML Content.</returns>
        [HttpGet]
        public ActionResult PrescriptionDispenseViewContentView(PrescriptionDispenseViewModel request)
        {
            // If invalid, return not found
            if (!ModelState.IsValid)
            {
                return this.HttpNotFound();
            }

            // Retrieve view
            var view = this.pcehrViewRepository.GetView(
                new CommonSchemas.PatientIdentifier.Mrn(request.PatientId, request.HospitalId, this.DefaultHospitalCodeSystem),
                this.GetCurrentUserDetails(),
                new PrescriptionAndDispenseViewRequest() { FromDate = request.FromDate.Value, ToDate = request.ToDate.Value });

            // If error retrieving view then return not found
            if (!view.IsSuccessful)
            {
                return this.HttpNotFound();
            }

            string renderedHtml = view.Data.TransformToHtml(
                    Properties.Resources.NEHTA_PCEHR_Prescription_and_Dispense_View_CDA_Stylesheet_1_1_0,
                    Url.Action("Document", new { request.HospitalId, request.PatientId, repositoryId = "Remove", documentId = "Remove" }).Replace("Remove/Remove", string.Empty), // TODO Better way to get root URL
                    request.PatientId,
                    request.FromDate.Value,
                    request.ToDate.Value,
                    request.GroupBy);

            // Return HTML
            return this.Content(renderedHtml);
        }

        #endregion Prescription Dispense View / Content

        #region Gain Access

        /// <summary>
        /// Shows the gain access form in embedded view. The embedded view of 
        /// Gain Access is given a separate action method so that the
        /// PcehrAccessFilter can redirect to the embedded or non-embedded
        /// view as appropriate.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="returnUrl">The original URL.</param>
        /// <returns>View or PartialView result.</returns>     
        [HttpGet]
        public ActionResult EmbeddedGainAccess(string hospitalId, string patientId, string returnUrl)
        {
            return this.GainAccess(hospitalId, patientId, returnUrl);
        }

        /// <summary>
        /// Gain access to the PCEHR for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="returnUrl">The original URL.</param>
        /// <returns>View or PartialView result.</returns>
        [HttpGet]
        public ActionResult GainAccess(string hospitalId, string patientId, string returnUrl)
        {
            // Create ViewModel:
            var m = new GainAccessViewModel() { HospitalId = hospitalId, PatientId = patientId, ReturnUrl = returnUrl };
            this.LoadCurrentContext(m);

            // Add message provided by PcehrAccessFilter / PcehrAccessManager if present.
            if (this.TempData.ContainsKey("PcehrAccessMessage") && (!string.IsNullOrEmpty((string)this.TempData["PcehrAccessMessage"])))
            {
                m.Messages.Add((string)this.TempData["PcehrAccessMessage"], MessageLevel.Warning);
            }

            // If Ajax request return a PartialView
            if (Request.IsAjaxRequest())
            {
                return this.PartialView(m);
            }

            return this.View("GainAccess", m);
        }

        /// <summary>
        /// Gain access to the PCEHR for a selected patient.
        /// </summary>
        /// <param name="hospitalId">Identifier of the hospital.</param>
        /// <param name="patientId">Identifier of the selected patient.</param>
        /// <param name="returnUrl">The original URL.</param>
        /// <param name="accessType">Access type selected by the user.</param>
        /// <param name="accessCode">Access code (if specified).</param>
        /// <returns>View result or Redirect to originally requested page.</returns>
        [HttpPost]
        public ActionResult GainAccess(string hospitalId, string patientId, string returnUrl, AccessType accessType, string accessCode = null)
        {
            // Create ViewModel:
            var m = new GainAccessViewModel() { HospitalId = hospitalId, PatientId = patientId, ReturnUrl = returnUrl, AccessType = accessType, AccessCode = accessCode };
            this.LoadCurrentContext(m);

            // Validate:
            if (accessType == AccessType.WithCode)
            {
                if (string.IsNullOrEmpty(accessCode))
                {
                    this.ModelState.AddModelError("AccessCode", "A non-empty Access Code must be provided when attempting to gain access using an access code.");
                }
            }

            if (this.ModelState.IsValid)
            {
                // Attempt to gain access:

                HIPS.Web.Components.ServiceModel.ServiceResponse<GainPcehrAccessResponse> result = null;

                switch (accessType)
                {
                    case AccessType.WithCode:
                        {
                            result = this.pcehrViewRepository.GainAccessWithCode(new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem), accessCode, this.GetCurrentUserDetails());
                            break;
                        }
                    case AccessType.Emergency:
                        {
                            result = this.pcehrViewRepository.GainAccessEmergency(new CommonSchemas.PatientIdentifier.Mrn(patientId, hospitalId, this.DefaultHospitalCodeSystem), this.GetCurrentUserDetails());
                            break;
                        }
                    default:
                        {
                            break;
                        }
                }

                if (result != null)
                {
                    if (result.IsSuccessful)
                    {
                        return this.Redirect(returnUrl);
                    }
                    else
                    {
                        string errorMessage = "Failed to gain access for the selected patient.";
                        // Log details:
                        Elmah.ErrorSignal.FromCurrentContext().Raise(new System.Exception(string.Format("{0} {1}", errorMessage, result.Messages.AsString())));
                        // Copy over response messages.
                        m.Messages.AddRange(ObjectMapper.Map<IEnumerable<ViewMessage>>(result.Messages));
                    }
                }
            }

            // Invalid outcome, redisplay view.
            return this.View(m);
        }

        #endregion Gain Access

        #endregion Actions

        #region Helpers

        /// <summary>
        /// Loads required context into the provided view model.
        /// </summary>
        /// <param name="model">View model to load context into.</param>
        private void LoadCurrentContext(PcehrViewModelBase model)
        {
            // Load current hospital.
            if (!string.IsNullOrEmpty(model.HospitalId))
            {
                var hospitals = ObjectMapper.Map<IEnumerable<HospitalViewModel>>(this.hospitalRepository.GetHospitals(this.DefaultHospitalCodeSystem), new Helpers.Mapping.Context.HospitalMappingContext(this.DefaultHospitalCodeSystem));

                if ((hospitals != null) && (hospitals.Count() > 0))
                {
                    model.CurrentHospital = hospitals.FirstOrDefault(h => h.Code == model.HospitalId);

                    // Embedded view needs to set the session so that IFrame inclusions don't redirect to Select HPO.
                    // Review this code when implementing facility security to avoid access to unauthorised hospitals.
                    this.SessionConfiguration.RepresentingHospital = ObjectMapper.Map<Hospital>(model.CurrentHospital);
                }
            }

            // Load current patient.
            if (!string.IsNullOrEmpty(model.PatientId))
            {
                model.CurrentPatient = ObjectMapper.Map<PatientViewModel>(this.patientRepository.GetPatientInHospital(new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem), this.GetCurrentUserDetails()));
            }

            // Load current document.
            if ((!string.IsNullOrEmpty(model.RepositoryId)) && (!string.IsNullOrEmpty(model.DocumentId)))
            {
                model.CurrentDocument = ObjectMapper.Map<DocumentViewModel>(this.pcehrViewRepository.GetDocumentMetaData(new CommonSchemas.PatientIdentifier.Mrn(model.PatientId, model.HospitalId, this.DefaultHospitalCodeSystem), model.RepositoryId, model.DocumentId, this.GetCurrentUserDetails()));
            }
        }

        #endregion Helpers

        #endregion Methods
    }
}