﻿#region Using Statements

using HIPS.CommonSchemas;
using HIPS.CommonSchemas.PatientIdentifier;
using HIPS.PcehrDataStore.Schemas.Enumerators;

#endregion Using Statements

namespace HIPS.Web.UI.Helpers
{
    /// <summary>
    /// Represents the result of a check for PCEHR access.
    /// </summary>
    public class PcehrAccessResult
    {
        #region Properties

        /// <summary>
        /// Gets or sets a value that indicates if access to the PCEHR is allowed.
        /// </summary>
        public bool AccessAllowed { get; private set; }

        /// <summary>
        /// Gets or sets a message providing further information on PCEHR access.
        /// </summary>
        public string AccessMessage { get; private set; }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="accessAllowed">A value that indicates if access to the PCEHR is allowed</param>
        /// <param name="accessMessage">A message providing further information on PCEHR access.</param>
        public PcehrAccessResult(bool accessAllowed, string accessMessage)
        {
            this.AccessAllowed = accessAllowed;
            this.AccessMessage = accessMessage;
        }

        #endregion Constructors
    }

    /// <summary>
    /// Supports checking access to the PCEHR and gaining access automatically if required and possible.
    /// </summary>
    public class PcehrAccessManager
    {
        #region Properties

        /// <summary>
        /// Gets or sets the patient repository to be used by this class.
        /// </summary>
        private HIPS.Web.ModelInterface.Common.IPatientRepository PatientRepository { get; set; }

        /// <summary>
        /// Gets or sets the PCEHR view repository to be used by this class.
        /// </summary>
        private HIPS.Web.ModelInterface.PcehrView.IPcehrViewRepository PcehrViewRepository { get; set; }

        #endregion Properties

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="patientRepository">Patient repository to be used by this class.</param>
        /// <param name="pcehrViewRepository">PCEHR view repository to be used by this class.</param>
        public PcehrAccessManager(HIPS.Web.ModelInterface.Common.IPatientRepository patientRepository, HIPS.Web.ModelInterface.PcehrView.IPcehrViewRepository pcehrViewRepository)
        {
            this.PatientRepository = patientRepository;
            this.PcehrViewRepository = pcehrViewRepository;
        }

        #endregion Constructors

        #region Methods

        /// <summary>
        /// Checks the PCEHR to determine if the specified hospital is allowed to access the specified patient's PCEHR.
        /// </summary>
        /// <param name="patientIdentifier">Identifier of the patient (MRN).</param>
        /// <param name="autoAccess">Boolean value indicating whether to automatically attempt to gain access if required.</param>
        /// <param name="operatingUser">User requesting the operation.</param>
        /// <returns>PcehrAccessResult indicating if the specified hospital is allowed to access the specified patient's PCEHR.</returns>
        public PcehrAccessResult AllowAccess(Mrn patientIdentifier, bool autoAccess, UserDetails operatingUser)
        {
            bool allowAccess = false;
            string message = string.Empty;

            //Get patient. Groan. Because we HAVE to pass the DOB. Groan some more.
            var patient = this.PatientRepository.GetPatientInHospital(patientIdentifier, operatingUser);
            if (patient == null)
            {
                return new PcehrAccessResult(true, message);
            }

            //Check access.
            var pcehrAccessStatus = this.PcehrViewRepository.IsPcehrAdvertised(
                patientIdentifier: patientIdentifier,
                dateOfBirth: patient.DateOfBirth,
                operatingUser: operatingUser);

            //Act upon access status.

            if (pcehrAccessStatus != null)
            {
                if (pcehrAccessStatus.IsSuccessful)
                {
                    switch (pcehrAccessStatus.Data.AccessCodeRequired)
                    {
                        case AccessCodeRequired.AccessGranted:
                            {
                                //We already have access!
                                allowAccess = true;
                                break;
                            }
                        case AccessCodeRequired.Unknown:
                        case AccessCodeRequired.WithoutCode:
                            {
                                if (autoAccess)
                                {
                                    // Attempt to gain access without code. If we still can't gain access we'll need to redirect.
                                    var pcehrAccessResult = this.PcehrViewRepository.GainAccessWithoutCode(
                                        patientIdentifier: patientIdentifier,
                                        operatingUser: operatingUser);

                                    if (pcehrAccessResult.IsSuccessful)
                                    {
                                        if (pcehrAccessResult.Data.AccessPermission == PcehrSchemas.GainPcehrAccessStatus.Permit)
                                        {
                                            allowAccess = true;
                                        }
                                        else
                                        {
                                            allowAccess = false;
                                            message = string.Format("An attempt to automatically gain access to the patient's PCEHR was denied. Please try an alternative means for gaining access. {0}", pcehrAccessStatus.Messages.AsString());
                                        }
                                    }
                                    else
                                    {
                                        allowAccess = false;
                                        message = string.Format("An attempt to automatically gain access to the patient's PCEHR failed. Please try an alternative means for gaining access. {0}", pcehrAccessStatus.Messages.AsString());
                                    }
                                }
                                else
                                {
                                    allowAccess = false;
                                }
                                break;
                            }
                        case AccessCodeRequired.WithCode:
                            {
                                //We can't gain access automatically, so we'll need to redirect.
                                allowAccess = false;
                                message = "A code is required to access the patient's PCEHR.";
                                break;
                            }
                    }
                }
                else
                {
                    allowAccess = false;
                    message = string.Format("Failed to retrieve the patient's PCEHR status. {0}", pcehrAccessStatus.Messages.AsString());
                }
            }
            else
            {
                allowAccess = false;
                message = "Failed to retrieve the patient's PCEHR status.";
            }

            return new PcehrAccessResult(allowAccess, message);
        }

        #endregion Methods
    }
}