using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.Model.AssistedRegistration;

namespace HIPS.Web.UI.ViewModels.AssistedRegistration
{
    /// <summary>
    /// Represents demographic/identifying information for a person.
    /// </summary>
    public class PersonDemographicViewModel
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="PersonDemographicViewModel" /> class.
        /// </summary>
        public PersonDemographicViewModel()
        {
            this.ShowDvaFileNumber = true;
            this.ShowIndigenousStatus = true;
            this.Sexes = Enumerable.Empty<SelectListItem>();
            this.IndigenousStatuses = Enumerable.Empty<SelectListItem>();
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets the MRN of the patient.
        /// </summary>
        public string Mrn { get; set; }

        /// <summary>
        /// Gets or sets the IHI of the patient.
        /// </summary>
        public string Ihi { get; set; }

        /// <summary>
        /// Gets or sets the Family name of the patient.
        /// </summary>
        [Required]
        [Display(Name = "Family Name")]
        public string FamilyName { get; set; }

        /// <summary>
        /// Gets or sets the Given names of the patient.
        /// </summary>
        [Required]
        [Display(Name = "Given Name")]
        public string GivenNames { get; set; }

        /// <summary>
        /// Gets or sets the date of birth of the patient.
        /// </summary>
        [Required]
        [Display(Name = "Date of Birth"), DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        /// <summary>
        /// Gets or sets Sex code.
        /// </summary>
        [Required]
        [Display(Name = "Sex")]
        public string SexCode { get; set; }

        /// <summary>
        /// Gets or sets Sex drop down.
        /// </summary>
        public IEnumerable<SelectListItem> Sexes { get; set; }

        /// <summary>
        /// Gets or sets Indigenous status code.
        /// </summary>
        [Display(Name = "Aboriginal or Torres Strait Islander Origin")]
        public string IndigenousStatusCode { get; set; }

        /// <summary>
        /// Gets or sets Indigenous status drop down.
        /// </summary>
        public IEnumerable<SelectListItem> IndigenousStatuses { get; set; }

        /// <summary>
        /// Gets or sets Medicare Number of the patient.
        /// </summary>
        [Display(Name = "Medicare Card Number")]
        public string MedicareNumber { get; set; }

        /// <summary>
        /// Gets or sets IRN of the patient.
        /// </summary>
        [Display(Name = "IRN")]
        public string MedicareNumberIrn { get; set; }

        /// <summary>
        /// Gets or sets DVA Number of the patient.
        /// </summary>
        [Display(Name = "DVA File Number")]
        public string DvaFileNumber { get; set; }

        #endregion Properties

        #region Calculated Properties

        /// <summary>
        ///  Gets a value indicating whether this view model represents known patient data,
        ///  including a validated IHI.
        /// </summary>
        public bool IsKnownPatient
        {
            get { return !string.IsNullOrWhiteSpace(this.Ihi); }
        }

        #endregion Calculated Properties

        #region View Logic Properties

        /// <summary>
        ///  Gets or sets a value indicating whether the DVA File Number is a valid identification option.
        /// </summary>
        public bool ShowDvaFileNumber { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether the Indigenous Status will be requested.
        /// </summary>
        public bool ShowIndigenousStatus { get; set; }

        #endregion View Logic Properties

        #region Data Source Methods

        /// <summary>
        /// Gets Indigenous status from list.
        /// </summary>
        /// <param name="indigenousStatuses">List of Indigenous status.</param>
        /// <returns>Indigenous status.</returns>
        public IndigenousStatus GetIndigenousStatus(List<IndigenousStatus> indigenousStatuses)
        {
            return indigenousStatuses.FirstOrDefault(i => i.Code == this.IndigenousStatusCode);
        }

        #endregion Data Source Methods

        #region Methods

        /// <summary>
        /// Creates a Person demographic.
        /// </summary>
        /// <param name="sexes">List of sex.</param>
        /// <returns>Person demographic.</returns>
        public PersonDemographic ToPersonDemographicModel(List<Sex> sexes)
        {
            Sex personSex = sexes.FirstOrDefault(s => s.Code == this.SexCode);
            return new PersonDemographic(this.FamilyName, this.GivenNames, this.DateOfBirth, personSex, this.MedicareNumber, this.MedicareNumberIrn, this.DvaFileNumber);
        }

        #endregion Methods
    }
}