using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.Model.AssistedRegistration;
using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.AssistedRegistration
{
    /// <summary>
    /// View Model used for the PCEHR Register view.
    /// </summary>
    public class RegisterViewModel
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="RegisterViewModel" /> class.
        /// </summary>
        public RegisterViewModel()
        {
            this.Applicant = new PersonDemographicViewModel();
            this.Representative = new PersonDemographicViewModel { ShowDvaFileNumber = false, ShowIndigenousStatus = false };
            this.MedicareConsents = new MedicareConsentViewModel();
            this.IdentityVerificationMethods = Enumerable.Empty<SelectListItem>();
            this.Messages = new ViewMessageList();
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets Hospital Id.
        /// </summary>
        public string HospitalId { get; set; }

        /// <summary>
        /// Gets or sets Applicant's Patient Master ID (used for retry submissions).
        /// </summary>
        public int? PatientMasterId { get; set; }

        /// <summary>
        /// Gets or sets the registering Applicant.
        /// </summary>
        public PersonDemographicViewModel Applicant { get; set; }

        /// <summary>
        /// Gets or sets the parent/guardian Representative for an Applicant who is a minor.
        /// Required for Applicants under the age of 14, optional for Applicants 14-18.
        /// </summary>
        public PersonDemographicViewModel Representative { get; set; }

        /// <summary>
        /// Gets or sets Medicare Consents.
        /// </summary>
        [Display(Name = "Medicare Access Consent")]
        public MedicareConsentViewModel MedicareConsents { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether Consents is declared.
        /// </summary>
        [Display(Name = "Consent")]
        public bool ConsentDeclared { get; set; }

        /// <summary>
        /// Gets or sets Phone number.
        /// </summary>
        public string Phone { get; set; }

        /// <summary>
        /// Gets or sets Email Address.
        /// </summary>
        public string Email { get; set; }

        /// <summary>
        /// Gets or sets Ivc delivery method.
        /// </summary>
        public string IvcDeliveryMethod { get; set; }

        /// <summary>
        /// Gets or sets Identity verification method.
        /// </summary>
        [Required]
        [Display(Name = "Identity Verification Method")]
        public string IdentityVerificationMethod { get; set; }

        /// <summary>
        /// Gets or sets Identity verification methods drop down.
        /// </summary>
        public IEnumerable<SelectListItem> IdentityVerificationMethods { get; set; }

        /// <summary>
        /// Gets or sets the list of messages associated with the view model.
        /// </summary>
        public ViewMessageList Messages { get; set; }


        #endregion Properties

        #region Methods

        /// <summary>
        /// Gets Identity verification method from list.
        /// </summary>
        /// <param name="identityVerificationMethods">List of Identity verification method.</param>
        /// <returns>Identity verification method.</returns>
        // TODO: General Design, these viewmodels shouldn't have model references. Refactor like HPII
        public IdentityVerificationMethod GetIdentityVerificationMethod(List<IdentityVerificationMethod> identityVerificationMethods)
        {
            return identityVerificationMethods.FirstOrDefault(i => i.Code == this.IdentityVerificationMethod);
        }

        /// <summary>
        /// Gets Ivc delivery method from list.
        /// </summary>
        /// <param name="ivcDeliveryMethods">List of Ivc delivery method.</param>
        /// <returns>Ivc delivery method.</returns>
        public IvcDeliveryMethodDecision GetIvcDeliveryMethodDecision(List<IvcDeliveryMethod> ivcDeliveryMethods)
        {
            var ivcMethod = ivcDeliveryMethods.FirstOrDefault(i => i.Code == this.IvcDeliveryMethod);

            if (ivcMethod == null)
            {
                return null;
            }

            // TODO - Better switch/case, confirm support, regex, not more than one email+phone, support new methods with destinations
            if (ivcMethod.Code == "email")
            {
                return new IvcDeliveryMethodDecision(ivcMethod, this.Email, this.Email);
            }
            else if (ivcMethod.Code == "sms")
            {
                return new IvcDeliveryMethodDecision(ivcMethod, this.Phone, this.Phone);
            }
            else
            {
                return new IvcDeliveryMethodDecision(ivcMethod);
            }
        }

        #endregion Methods

        public bool HasPcehr { get; set; }
    }
}