﻿using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Web;
using System.Web.Mvc;

using HIPS.Web.Model;

namespace HIPS.Web.UI.ViewModels.DocumentUpload
{
    /// <summary>
    /// View model for previewing discharge summary document.
    /// </summary>
    public class CreateDocumentViewModel : DocumentUploadViewModelBase
    {
        #region Constructors

        /// <summary>
        /// Initialises a new instance of the <see cref="CreateDocumentViewModel" /> class.
        /// </summary>
        public CreateDocumentViewModel()
        {
            this.CompletionDate = DateTime.Now;
        }

        #endregion Constructors

        #region Properties

        /// <summary>
        /// Gets or sets current user.
        /// </summary>
        public UserViewModel CurrentUser { get; set; }

        /// <summary>
        /// Gets or sets responsible user.
        /// </summary>
        public UserViewModel ResponsibleUser { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether if the clinical document is confirmed.
        /// </summary>
        [Required]
        [Display(Name = "Confirm Upload Clinical Document")]
        public bool Confirm { get; set; }

        /// <summary>
        /// Gets or sets completion date and time of the document.
        /// </summary>
        [Required]
        [Display(Name = "Date and Time of Completion")]
        public DateTime CompletionDate { get; set; }

        /// <summary>
        /// Gets or sets Source Document Status Id.
        /// </summary>
        [Required]
        [Display(Name = "Source Document Status")]
        public int SourceDocumentStatusId { get; set; }

        /// <summary>
        /// Gets or sets Source Document status dropdown.
        /// </summary>
        public IEnumerable<SelectListItem> SourceDocumentStatus { get; set; }

        /// <summary>
        /// Gets or sets Mode of Separation Id.
        /// </summary>
        [Required]
        [Display(Name = "Mode of Separation")]
        public int ModesOfSeparationId { get; set; }

        /// <summary>
        /// Gets or sets Mode of Separation dropdown.
        /// </summary>
        public IEnumerable<SelectListItem> ModesOfSeparation { get; set; }

        /// <summary>
        /// Gets or sets Clinical specialty.
        /// </summary>
        [Required]
        [Display(Name = "Clinical Specialty")]
        public string ClinicalSpecialtyCode { get; set; }

        /// <summary>
        /// Gets or sets Clinical Specialty dropdown.
        /// </summary>
        public IEnumerable<SelectListItem> ClinicalSpecialty { get; set; }

        /// <summary>
        /// Gets or sets Discharge summary attachment.
        /// </summary>
        [Required(ErrorMessage = "Discharge Summary attachment is required.")]
        [Display(Name = "Discharge Summary attachment")]
        [HIPS.Web.Components.Validation.FileExtension(Extensions = ".pdf")]
        [HIPS.Web.Components.Validation.FileSize]
        public HttpPostedFileBase DischargeSummaryAttachment { get; set; }

        /// <summary>
        /// Gets or sets Image attachments.
        /// </summary>
        [Display(Name = "Image attachments")]
        [HIPS.Web.Components.Validation.FileExtension(Extensions = ".png|.gif|.jpeg|.jpg|.tiff")]
        [HIPS.Web.Components.Validation.FileSize]
        public IEnumerable<HttpPostedFileBase> ImageAttachments { get; set; }

        #endregion Properties

        #region Methods

        /// <summary>
        /// Returns description of mode of separation.
        /// </summary>
        /// <param name="modeOfSeparation">Mode of separation.</param>
        /// <returns>Mode of separation description.</returns>
        public string SetModeOfSeparationDescription(ModeOfSeparation modeOfSeparation)
        {
            switch (modeOfSeparation)
            {
                case ModeOfSeparation.AcuteHospital:
                    return "Acute Hospital";

                case ModeOfSeparation.AdministrativeDischarge:
                    return "Administrative Discharge";

                case ModeOfSeparation.AdministrativeFromLeave:
                    return "Administrative From Leave";

                case ModeOfSeparation.AgedCareService:
                    return "Aged Care Service";

                case ModeOfSeparation.Deceased:
                    return "Deceased";

                case ModeOfSeparation.Home:
                    return "Home";

                case ModeOfSeparation.None:
                    return "None";

                case ModeOfSeparation.OtherHealthService:
                    return "Other Health Service";

                case ModeOfSeparation.PsychiatricCare:
                    return "Psychiatric Care";

                case ModeOfSeparation.SelfDischarge:
                    return "Self Discharge";

                default:
                    return "None";
            }
        }

        #endregion Methods
    }
}