﻿using System;
using System.ComponentModel.DataAnnotations;

using HIPS.Web.Model;

namespace HIPS.Web.UI.ViewModels.DocumentUpload
{
    /// <summary>
    /// View model for a single episode of care when the patient was admitted to hospital.
    /// </summary>
    public class EpisodeViewModel
    {
        #region Properties

        /// <summary>
        /// Gets or sets the visit number that identifies this episode.
        /// </summary>
        public string SourceSystemEpisodeId { get; set; }

        /// <summary>
        /// Gets or sets the Set ID of the document, which is common to all versions of the document.
        /// </summary>
        public string SourceSystemSetId { get; set; }

        /// <summary>
        /// Gets or sets the date and time when the patient was admitted.
        /// </summary>
        [Display(Name = "Admission Date")]
        [DisplayFormat(DataFormatString = "{0:dd/MM/yyyy HH:mm}")]
        public DateTime AdmissionDate { get; set; }

        /// <summary>
        /// Gets or sets the date and time when the patient was discharged, or null if the patient is still in hospital.
        /// </summary>
        [Required]
        [Display(Name = "Discharge Date")]
        [DisplayFormat(DataFormatString = "{0:dd/MM/yyyy HH:mm}", ApplyFormatInEditMode = true)]
        public DateTime? DischargeDate { get; set; }

        /// <summary>
        /// Gets or sets the reason why the patient was admitted.
        /// </summary>
        [Display(Name = "Admission Reason")]
        public string AdmissionReason { get; set; }

        /// <summary>
        /// Gets or sets the identifier of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderIdentifier { get; set; }

        /// <summary>
        /// Gets or sets the family name of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderFamilyName { get; set; }

        /// <summary>
        /// Gets or sets the given name of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderGivenNames { get; set; }

        /// <summary>
        /// Gets or sets the name title of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderTitle { get; set; }

        /// <summary>
        /// Gets or sets the name suffix of the responsible provider for this episode.
        /// </summary>
        public string ResponsibleProviderSuffix { get; set; }

        /// <summary>
        /// Gets or sets the Local Document status for this episode.
        /// </summary>
        public LocalDocumentStatus Status { get; set; }

        /// <summary>
        /// Gets or sets the ward component of the assigned location for the patient within this episode.
        /// </summary>
        public string Ward { get; set; }

        /// <summary>
        /// Gets or sets the room component of the assigned location for the patient within this episode.
        /// </summary>
        public string Room { get; set; }

        /// <summary>
        /// Gets or sets the bed component of the assigned location for the patient within this episode.
        /// </summary>
        public string Bed { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether consent has been withdrawn.
        /// </summary>
        /// <value>
        ///   <c>true</c> if consent withdrawn; otherwise, <c>false</c>.
        /// </value>
        public bool ConsentWithdrawn { get; set; }

        /// <summary>
        /// Gets or sets a value indicating whether if the patient has PCEHR.
        /// </summary>
        public bool HasPcehr { get; set; }

        #region Calculated Properties

        /// <summary>
        /// Gets the name of the responsible provider for this episode.
        /// </summary>
        [Display(Name = "Responsible Provider")]
        public string ResponsibleProviderName
        {
            get
            {
                return string.Format("{0} {1} {2} {3}", this.ResponsibleProviderTitle, this.ResponsibleProviderGivenNames, this.ResponsibleProviderFamilyName, this.ResponsibleProviderSuffix);
            }
        }

        /// <summary>
        /// Gets the location (ward:room:bed) of the patient's most recent episode.
        /// </summary>
        public string Location
        {
            get
            {
                return string.Format("{0}:{1}:{2}", this.Ward, this.Room, this.Bed);
            }
        }

        /// <summary>
        /// Gets the document status as a string.
        /// </summary>
        [Display(Name = "Upload Status")]
        public string DocumentStatus
        {
            get
            {
                if (this.ConsentWithdrawn)
                {
                    return "Consent Withdrawn";
                }
                else
                {
                    switch (this.Status)
                    {
                        case LocalDocumentStatus.Unknown:
                            return "None";

                        case LocalDocumentStatus.PendingUpload:
                            return "Pending Upload";

                        case LocalDocumentStatus.Uploaded:
                            return "Uploaded";

                        case LocalDocumentStatus.FailedUpload:
                            return "Failed Upload";

                        case LocalDocumentStatus.PendingRemove:
                            return "Pending Remove";

                        case LocalDocumentStatus.Removed:
                            return "Removed";

                        case LocalDocumentStatus.FailedRemove:
                            return "Failed Remove";

                        default:
                            return "Unknown";
                    }
                }
            }
        }

        #endregion Calculated Properties

        #endregion Properties
    }
}