﻿using System.Collections.Generic;
using AutoMapper;
using HIPS.ConsentSchemas;
using HIPS.PatientSchemas;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.ServiceContracts.Patient.Message;

using DataSchemas = HIPS.PcehrDataStore.Schemas;
using ServiceSchemas = HIPS.PatientSchemas;

namespace HIPS.AppServer.ServiceHost.Mapping.Profiles
{
    /// <summary>
    /// AutoMapper mapping profile for the Patient entity.
    /// </summary>
    internal class PatientProfile : Profile
    {
        #region Methods

        /// <summary>
        /// Configures the maps available as part of this mapping profile.
        /// </summary>
        protected override void Configure()
        {
            //Data --> DTO:
            this.CreateMap<PatientInHospitalEpisode, PatientEpisode>();

            this.CreateMap<DataSchemas.PatientInHospital, ServiceSchemas.PatientInHospital>()
                .ForMember(dest => dest.ParticipationStatus, opt => opt.ResolveUsing<ParticipationStatusResolver>().FromMember(src => src))
                .ForMember(dest => dest.RegisteredSex, opt => opt.ResolveUsing<SexEnumeratorResolver>().FromMember(src => src.RegisteredSexId))
                .ForMember(dest => dest.CurrentSex, opt => opt.ResolveUsing(src => (SexEnumerator)src.CurrentSexId))
                .ForMember(dest => dest.IhiStatus, opt => opt.ResolveUsing(src => (src.IhiStatusId == 0 ? IhiStatus.Unknown : (IhiStatus)src.IhiStatusId)))
                .ForMember(dest => dest.IhiRecordStatus, opt => opt.ResolveUsing(src => (src.IhiRecordStatusId == 0 ? IhiRecordStatus.Unknown : (IhiRecordStatus)src.IhiRecordStatusId)));

            this.CreateMap<DataSchemas.PatientInHospital, ServiceSchemas.AdmittedPatient>()
                .ForMember(dest => dest.ParticipationStatus, opt => opt.ResolveUsing<ParticipationStatusResolver>().FromMember(src => src))
                .ForMember(dest => dest.RegisteredSex, opt => opt.ResolveUsing<SexEnumeratorResolver>().FromMember(src => src.RegisteredSexId))
                .ForMember(dest => dest.CurrentSex, opt => opt.ResolveUsing(src => (SexEnumerator)src.CurrentSexId))
                .ForMember(dest => dest.IhiStatus, opt => opt.ResolveUsing(src => (src.IhiStatusId == 0 ? IhiStatus.Unknown : (IhiStatus)src.IhiStatusId)))
                .ForMember(dest => dest.IhiRecordStatus, opt => opt.ResolveUsing(src => (src.IhiRecordStatusId == 0 ? IhiRecordStatus.Unknown : (IhiRecordStatus)src.IhiRecordStatusId)));

            this.CreateMap<EpisodePatientExtendedDetails, EpisodePatientDetailsResponse>()
                .ForMember(dest => dest.CurrentSex, opt => opt.ResolveUsing(src => (SexEnumerator)src.CurrentSexId));

            this.CreateMap<PatientDisclosureDetails, PatientDisclosureDetailsResponse>()
                .ForMember(dest => dest.AdmittedPatient, opt => opt.MapFrom(src => src.PatientInHospital));

            this.CreateMap<List<DataSchemas.PatientInHospitalEpisode>, ListPatientEpisodesInHospitalResponse>()
                .ForMember(dest => dest.Data, opt => opt.MapFrom(src => src));

            this.CreateMap<DataSchemas.PatientInHospitalEpisode, HIPS.ServiceContracts.Common.DTO.PatientEpisode>();
        }

        /// <summary>
        /// Resolves the value of the ParticipationStatus
        /// </summary>
        internal class ParticipationStatusResolver : ValueResolver<DataSchemas.PatientInHospital, ParticipationStatus>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override ParticipationStatus ResolveCore(DataSchemas.PatientInHospital source)
            {
                ParticipationStatus returnStatus;

                if (!string.IsNullOrEmpty(source.Ihi) && (source.IhiStatusId == 0 ? IhiStatus.Unknown : (IhiStatus)source.IhiStatusId) == IhiStatus.Active)
                {
                    if (source.PcehrAdvertised.HasValue && source.PcehrAdvertised.Value)
                    {
                        returnStatus = ParticipationStatus.PcehrAdvertised;
                    }
                    else if (source.PcehrDisclosed.HasValue && source.PcehrDisclosed.Value)
                    {
                        returnStatus = ParticipationStatus.RequestedUpload;
                    }
                    else
                    {
                        returnStatus = ParticipationStatus.PcehrNotAdvertised;
                    }
                }
                else
                {
                    returnStatus = ParticipationStatus.NoValidIhi;
                }

                return returnStatus;
            }

            #endregion Methods
        }

        /// <summary>
        /// Resolves the value of the SexEnumerator
        /// </summary>
        internal class SexEnumeratorResolver : ValueResolver<int?, SexEnumerator>
        {
            #region Methods

            /// <summary>
            /// Resolve the required value.
            /// </summary>
            /// <param name="source">ResolutionResult representing the source.</param>
            /// <returns>ResolutionResult representing the destination.</returns>
            protected override SexEnumerator ResolveCore(int? source)
            {
                SexEnumerator returnSex;

                if (source.HasValue)
                {
                    returnSex = (SexEnumerator)source;
                }
                else
                {
                    returnSex = SexEnumerator.NotStatedOrInadequatelyDescribed;
                }

                return returnSex;
            }

            #endregion Methods
        }

        /// <summary>
        /// Returns the name of the mapping profile.
        /// </summary>
        public override string ProfileName
        {
            get { return this.GetType().Name; }
        }

        #endregion Methods
    }
}