﻿using System;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using Nehta.VendorLibrary.Common;

namespace HIPS.CommonBusinessLogic.Ihi
{
    public static class Helpers
    {
        /// <summary>
        /// Clones the specified source.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="source">The source.</param>
        /// <returns></returns>
        public static T Clone<T>(this T source)
        {
            if (!typeof(T).IsSerializable)
            {
                throw new ArgumentException("The type must be serializable.", "source");
            }

            // Don't serialize a null object, simply return the default for that object
            if (Object.ReferenceEquals(source, null))
            {
                return default(T);
            }

            IFormatter formatter = new BinaryFormatter();
            Stream stream = new MemoryStream();
            using (stream)
            {
                formatter.Serialize(stream, source);
                stream.Seek(0, SeekOrigin.Begin);
                return (T)formatter.Deserialize(stream);
            }
        }

        /// <summary>
        /// Extracts the ihi from the fully qualified string.
        /// </summary>
        /// <param name="ihi">The ihi.</param>
        /// <returns></returns>
        public static string ExtractIhi(string ihi)
        {
            if (string.IsNullOrEmpty(ihi))
            {
                return string.Empty;
            }
            string result = ihi;
            if (ihi != null && ihi.StartsWith(HIQualifiers.IHIQualifier))
            {
                result = ihi.Substring(HIQualifiers.IHIQualifier.Length);
            }

            return result;
        }

        /// <summary>
        /// Gets the IHI record status value from the string passed in back to an integer used in code.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        public static int GetIhiRecordStatusValue(string name)
        {
            int result = -1;
            string check = name.ToUpper();
            if (check == HelperResource.IhiRecordStatusProvisional)
            {
                result = (int)IhiRecordStatus.Provisional;
            }
            else if (check == HelperResource.IhiRecordStatusUnverified)
            {
                result = (int)HIPS.PcehrDataStore.Schemas.Enumerators.IhiRecordStatus.Unverified;
            }
            else if (check == HelperResource.IhiRecordStatusVerified)
            {
                result = (int)IhiRecordStatus.Verified;
            }
            else
            {
                result = (int)IhiRecordStatus.Unverified;
            }
            return result;
        }

        /// <summary>
        /// Gets the IHI status value from the string passed in back to an integer used in code.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        public static int GetIhiStatusValue(string name)
        {
            int result = -1;
            string check = name.ToUpper();
            if (check == HelperResource.IhiStatusActive)
            {
                result = (int)IhiStatus.Active;
            }
            else if (check == HelperResource.IhiStatusDeceased)
            {
                result = (int)IhiStatus.Deceased;
            }
            else if (check == HelperResource.IhiStatusExpired)
            {
                result = (int)IhiStatus.Expired;
            }
            else if (check == HelperResource.IhiStatusResolved)
            {
                result = (int)IhiStatus.Resolved;
            }
            else if (check == HelperResource.IhiStatusRetired)
            {
                result = (int)IhiStatus.Retired;
            }
            else
            {
                result = (int)IhiStatus.Unknown;
            }
            return result;
        }

        /// <summary>
        /// Inserts the ihi identifier in front of the IHI if required
        /// </summary>
        /// <param name="ihi">The ihi.</param>
        /// <returns></returns>
        public static string InsertIhiIdentifier(string ihi)
        {
            string result = ihi;
            if (ihi != null && !ihi.StartsWith(HIQualifiers.IHIQualifier))
            {
                result = HIQualifiers.IHIQualifier + ihi;
            }

            return result;
        }

        /// <summary>
        /// Updates the specified target item.
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="targetItem">The target item.</param>
        /// <param name="sourceItem">The source item.</param>
        public static void Update<T>(T targetItem, T sourceItem)
        {
            PropertyInfo[] fromFields = typeof(T).GetProperties();
            for (int count = 0; count < fromFields.Length; count++)
            {
                PropertyInfo fromField = (PropertyInfo)fromFields[count];
                if (fromField.CanWrite)
                {
                    object value = fromField.GetValue(sourceItem, null);
                    fromField.SetValue(targetItem, value, null);
                }
            }
        }
    }
}