﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlTypes;
using System.Net;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using HIPS.CommonBusinessLogic.Pcehr;
using HIPS.CommonSchemas;
using HIPS.Configuration;
using HIPS.IhiSchemas.Exceptions;
using HIPS.IhiSchemas.Schemas;
using HIPS.PcehrDataStore.DataAccess;
using HIPS.PcehrDataStore.Schemas;
using HIPS.PcehrDataStore.Schemas.Enumerators;
using HIPS.PcehrSchemas;
using HIPS.PcehrSchemas.Exceptions;
using nehta.mcaR3.ConsumerSearchIHI;
using Nehta.VendorLibrary.Common;
using Nehta.VendorLibrary.HI;

namespace HIPS.CommonBusinessLogic.Ihi
{
    public static class Search
    {
        #region Constants

        private const int MAX_NAME_SIZE = 40;

        #endregion Constants

        /// <summary>
        /// Performs a search by the Medicare or DVA number or validate an IHI.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <param name="resultMessage">The oldItem message.</param>
        /// <returns></returns>
        static public bool ByNumber(Hospital hospital, PatientMaster patientMaster, IhiSearchCriteria ihiSearchCriteria, out string resultMessage, out searchIHIResponse ihiResponse, UserDetails user, bool checkPcehr = false)
        {
            try
            {
                ihiResponse = null;
                if (!string.IsNullOrEmpty(ihiSearchCriteria.MedicareNumber))
                {
                    ihiSearchCriteria.IsMedicareNumberValid = Medicare.Validate(ihiSearchCriteria.MedicareNumber);
                }

                if (ihiSearchCriteria.IsMedicareNumberValid.HasValue && !ihiSearchCriteria.IsMedicareNumberValid.Value)
                {
                    IhiLookupAlertBl.Insert(ihiSearchCriteria.PatientMasterId, ConstantsResource.InvalidMedicareCardNumber, null, null);
                    resultMessage = ConstantsResource.InvalidMedicareCardNumber;
                    return false;
                }

                if (patientMaster.IhiStatusId == (int)IhiStatus.MergeConflict)
                {
                    resultMessage = ConstantsResource.NoSearchingWhileInMergeConflict;
                    return false;
                }

                if (!User.ValidateUser(user))
                {
                    resultMessage = ConstantsResource.MissingUserValues;
                    return false;
                }

                if (ValidateSearchCriteria(ihiSearchCriteria))
                {
                    try
                    {
                        return PerformSearch(hospital, patientMaster, ihiSearchCriteria, out resultMessage, out ihiResponse, user, checkPcehr);
                    }
                    catch (Exception e)
                    {
                        resultMessage = e.Message;
                        ihiSearchCriteria.Ihi = null;
                        ihiSearchCriteria.PcehrExists = null;
                        throw e;
                    }
                }
                else
                {
                    resultMessage = ConstantsResource.MissingSearchValues;
                    return false;
                }
            }
            catch (Exception ex)
            {
                resultMessage = ex.Message;
                throw ex;
            }
        }

        /// <summary>
        /// Audits the specified ihi response.
        /// </summary>
        /// <param name="ihiResponse">The ihi response.</param>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <param name="user">The user.</param>
        /// <param name="errorMessage">The error message.</param>
        /// <param name="counter">The counter.</param>
        /// <param name="response">The response.</param>
        /// <param name="request">The request.</param>
        private static void Audit(searchIHIResponse ihiResponse, IhiSearchCriteria ihiSearchCriteria, UserDetails user, Hospital hospital, string errorMessage, int counter, string response, string request)
        {
            IhiLookupAudit ihiLookupAudit = new IhiLookupAudit();

            if (ihiResponse != null)
            {
                ihiLookupAudit.IhiNumber = ihiSearchCriteria.Ihi;
                if (!string.IsNullOrEmpty(ihiResponse.searchIHIResult.ihiNumber))
                {
                    ihiLookupAudit.IhiRecordStatus = Helpers.GetIhiRecordStatusValue(ihiResponse.searchIHIResult.ihiRecordStatus.ToString());
                    ihiLookupAudit.IhiStatus = Helpers.GetIhiStatusValue(ihiResponse.searchIHIResult.ihiStatus.ToString());
                }
            }
            else
            {
                ihiLookupAudit.Message = errorMessage;

                // Set to unknown because FK relationship is now enforced by database, default 0 is not acceptable.
                ihiLookupAudit.IhiRecordStatus = (int)IhiRecordStatus.Unknown;
                ihiLookupAudit.IhiStatus = (int)IhiStatus.Unknown;
            }
            ihiLookupAudit.MedicareNumber = string.Format("{0}{1}", ihiSearchCriteria.MedicareNumber, ihiSearchCriteria.MedicareIrn);
            ihiLookupAudit.PatientMasterId = ihiSearchCriteria.PatientMasterId;
            ihiLookupAudit.SexID = (int)ihiSearchCriteria.Names[counter].Sex;

            if (ihiSearchCriteria.Names[counter].DateOfBirth.HasValue)
            {
                ihiLookupAudit.DateOfBirth = ihiSearchCriteria.Names[counter].DateOfBirth.Value;
            }
            else
            {
                ihiLookupAudit.DateOfBirth = ihiSearchCriteria.DateOfBirth.Value;
            }

            ihiLookupAudit.DvaNumber = ihiSearchCriteria.DvaNumber;
            ihiLookupAudit.FamilyName = LimitSize(ihiSearchCriteria.Names[counter].FamilyName);
            ihiLookupAudit.GivenName = LimitSize(ihiSearchCriteria.Names[counter].GivenName);
            ihiLookupAudit.HpiI = user.HpiI;
            ihiLookupAudit.HpiO = hospital.HpiO;
            if (!string.IsNullOrEmpty(user.Login))
            {
                ihiLookupAudit.Operator = user.Domain + @"/" + user.Login + "(" + user.Name + ")";
            }
            else
            {
                ihiLookupAudit.Operator = user.Name;
            }
            ihiLookupAudit.Request = request;
            ihiLookupAudit.Response = response;
            IhiLookupAuditDl dataAccess = new IhiLookupAuditDl(user);
            dataAccess.Insert(ihiLookupAudit);
        }

        /// <summary>
        /// Checks whether the IHI search response contains a service message with one of
        /// the codes in the given list.
        /// </summary>
        /// <param name="ihiResponse">The response received from the IHI search</param>
        /// <param name="codes">A list of codes.</param>
        /// <returns>Whether the service messages included one from the list.</returns>
        private static bool DidReturnMessageCode(searchIHIResponse ihiResponse, List<string> codes)
        {
            if (ihiResponse != null
                && ihiResponse.searchIHIResult != null
                && ihiResponse.searchIHIResult.serviceMessages != null
                && ihiResponse.searchIHIResult.serviceMessages.serviceMessage != null)
            {
                foreach (ServiceMessageType message in ihiResponse.searchIHIResult.serviceMessages.serviceMessage)
                {
                    if (codes.Contains(message.code))
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        /// <summary>
        /// Gets the QualifiedId for the HPI-O element in the HI Service B2B Header.
        ///
        /// If the certificate belongs to a healthcare provider organisation, the HPI-O
        /// must not be provided in the header.
        ///
        /// If the certificate belongs to a contracted service provider, the HPI-O
        /// of the healthcare provider organisation (that the CSP is acting on behalf
        /// of) must be provided in the header.
        /// </summary>
        /// <returns>HPI-O element or null</returns>
        private static QualifiedId GetHpioQualifiedId(UserDetails userDetails, Hospital hospital)
        {
            // 1. If the hospital has the “HiCsp” flag set, return the HPI-O as a qualified ID.
            if (hospital.HiCsp)
            {
                QualifiedId hpio = new QualifiedId()
                {
                    id = hospital.HpiO,
                    qualifier = ConfigurationManager.AppSettings["IhiHpioQualifier"].ToString()
                };
                return hpio;
            }
            else
            {
                // 2. If the hospital does not have the “HiCsp” flag set, return null.
                return null;
            }
        }

        /// <summary>
        /// Gets the certificate.
        /// </summary>
        /// <returns></returns>
        private static X509Certificate2 GetMedicareCertificate(Hospital hospital)
        {
            string serialNumber = hospital.HiCertSerial;

            //string serialNumber = ConfigurationManager.AppSettings["IhiLookupCertificate"];
            if (string.IsNullOrEmpty(serialNumber))
            {
                throw new Exception("IHI certificate Serial has not been defined the database");
            }

            // Obtain the certificate by serial number
            X509Certificate2 tlsCert = X509CertificateUtil.GetCertificate(
                serialNumber,
                X509FindType.FindBySerialNumber,
                StoreName.My,
                StoreLocation.LocalMachine,
                false
            );
            if (tlsCert == null || tlsCert.PrivateKey == null)
            {
                throw new Exception("IHI certificate could not be loaded from store");
            }
            return tlsCert;
        }

        /// <summary>
        /// Gets the product.
        /// </summary>
        /// <returns></returns>
        private static ProductType GetProduct()
        {
            // Set up client product information (PCIN)
            // Values below should be provided by Medicare
            ProductType product = new ProductType()
            {
                platform = System.Environment.OSVersion.ToString(),
                productName = ConfigurationManager.AppSettings["IhiProductName"],                              // Provided by Medicare
                productVersion = ConfigurationManager.AppSettings["IhiProductVersion"],
                vendor = new QualifiedId()
                {
                    id = ConfigurationManager.AppSettings["IhiVendorId"],                                      // Provided by Medicare
                    qualifier = ConfigurationManager.AppSettings["IhiVendorQualifier"]
                }
            };
            return product;
        }

        /// <summary>
        /// Gets the search details which depend on the pass through in the code
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <param name="passCounter">The pass counter.</param>
        /// <returns></returns>
        private static searchIHI GetSearchDetails(IhiSearchCriteria ihiSearchCriteria, int passCounter)
        {
            searchIHI search = new searchIHI();

            search.dvaFileNumber = ihiSearchCriteria.DvaNumber;
            search.familyName = LimitSize(ihiSearchCriteria.Names[passCounter].FamilyName);
            search.givenName = LimitSize(ihiSearchCriteria.Names[passCounter].GivenName);
            search.dateOfBirth = (DateTime)(ihiSearchCriteria.Names[passCounter].DateOfBirth == null ? ihiSearchCriteria.DateOfBirth.Value : ihiSearchCriteria.Names[passCounter].DateOfBirth);

            if (!string.IsNullOrEmpty(ihiSearchCriteria.MedicareNumber))
            {
                search.medicareCardNumber = ihiSearchCriteria.MedicareNumber;
                search.medicareIRN = ihiSearchCriteria.MedicareIrn;
            }
            else
            {
                search.dvaFileNumber = ihiSearchCriteria.DvaNumber;
            }
            if (!string.IsNullOrEmpty(ihiSearchCriteria.Ihi))
            {
                search.ihiNumber = ihiSearchCriteria.Ihi;
            }
            switch (ihiSearchCriteria.Names[passCounter].Sex)
            {
                case SexEnumerator.Female:
                    search.sex = SexType.F;
                    break;

                case SexEnumerator.Male:
                    search.sex = SexType.M;
                    break;

                case SexEnumerator.IntersexOrIndeterminate:
                    search.sex = SexType.I;
                    break;

                default:
                    search.sex = SexType.N;
                    break;
            }
            return search;
        }

        /// <summary>
        /// Gets the client.
        /// </summary>
        /// <param name="tlsCert">The TLS certificate.</param>
        /// <param name="signingCert">The signing certificate.</param>
        /// <param name="product">The product.</param>
        /// <param name="user">The user.</param>
        /// <param name="hpio">The hpio.</param>
        /// <returns></returns>
        private static ConsumerSearchIHIClient GetTheClient(X509Certificate2 tlsCert, X509Certificate2 signingCert, ProductType product, QualifiedId user, QualifiedId hpio)
        {
            // Instantiate the client
            ConsumerSearchIHIClient client = new ConsumerSearchIHIClient(
                new Uri(ConfigurationManager.AppSettings["HiServiceUrl"].ToString()),
                product,
                user,
                hpio,
                signingCert,
                tlsCert);

            System.Reflection.FieldInfo field = client.GetType().GetField("searchIhiClient", System.Reflection.BindingFlags.Instance | System.Reflection.BindingFlags.NonPublic);
            ConsumerSearchIHIPortTypeClient port = field.GetValue(client) as ConsumerSearchIHIPortTypeClient;
            System.ServiceModel.Channels.CustomBinding binding = port.Endpoint.Binding as System.ServiceModel.Channels.CustomBinding;
            System.ServiceModel.Channels.HttpsTransportBindingElement https = binding.Elements[1] as System.ServiceModel.Channels.HttpsTransportBindingElement;

            //Set the connection timeout for the ConsumerSearchIHIClient service
            binding.OpenTimeout = TimeSpan.FromSeconds(Settings.Instance.IhiSearchTimeoutSeconds);
            binding.ReceiveTimeout = TimeSpan.FromSeconds(Settings.Instance.IhiSearchTimeoutSeconds);
            binding.SendTimeout = TimeSpan.FromSeconds(Settings.Instance.IhiSearchTimeoutSeconds);

            if (Settings.Instance.AvoidProxy)
            {
                https.UseDefaultWebProxy = false;
            }

            //Added in correct validation
            ServicePointManager.ServerCertificateValidationCallback += DocumentHelper.ValidateServiceCertificate;
            return client;
        }

        /// <summary>
        /// Gets the user.
        /// </summary>
        /// <returns></returns>
        private static QualifiedId GetUserQualifiedId(UserDetails userDetails)
        {
            QualifiedId qualifiedId;
            switch (userDetails.Role)
            {
                case UserRole.ProviderIndividual:
                    qualifiedId = new QualifiedId()
                    {
                        id = userDetails.HpiI,
                        qualifier = ConfigurationManager.AppSettings["IhiUserQualifierProviderIndividual"].ToString()
                    };
                    break;

                case UserRole.InteractiveUser:
                    qualifiedId = new QualifiedId()
                    {
                        id = userDetails.Login,
                        qualifier = string.Format(ConfigurationManager.AppSettings["IhiUserQualifierHiUser"].ToString(), userDetails.Domain)
                    };
                    break;

                case UserRole.AuthorisedEmployee:
                    qualifiedId = new QualifiedId()
                    {
                        id = userDetails.AuthorisedEmployeeUserId,
                        qualifier = ConfigurationManager.AppSettings["IhiUserQualifierAuthorisedEmployee"].ToString()
                    };
                    break;

                default:
                    throw new Exception("Unhandled User Role");
            }
            return qualifiedId;
        }

        /// <summary>
        /// Checks whether the IHI search response contains a service message
        /// that indicates the IHI was resolved.
        ///
        /// The particular codes that are checked for are "01606", "01611" and
        /// "01612", as defined by DHS Medicare document TECH.SIS.HI.06
        /// Section 11, "Information, Warning and Error Messages".
        /// </summary>
        /// <param name="ihiResponse">The response received from the IHI search</param>
        /// <returns>True if the IHI was resolved, otherwise false.</returns>
        private static bool IsResolved(searchIHIResponse ihiResponse)
        {
            List<string> resolvedCodes = new List<string> { "01606", "01611", "01612" };
            return DidReturnMessageCode(ihiResponse, resolvedCodes);
        }

        /// <summary>
        /// Checks whether the IHI service returned a message saying that
        /// the IHI had a retired status.
        /// </summary>
        /// <param name="ihiResponse"></param>
        /// <returns></returns>
        private static bool IsRetired(searchIHIResponse ihiResponse)
        {
            List<string> retiredCodes = new List<string> { "01614" };
            return DidReturnMessageCode(ihiResponse, retiredCodes);
        }

        /// <summary>
        /// Limits the length of a name as the web call will fail if it is too long.
        /// </summary>
        /// <param name="name">The name.</param>
        /// <returns></returns>
        private static string LimitSize(string name)
        {
            string result = name;
            if (result.Length > MAX_NAME_SIZE)
            {
                result = result.Substring(0, MAX_NAME_SIZE);
            }
            return result;
        }

        /// <summary>
        /// Performs the search.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <returns></returns>
        static private bool PerformSearch(Hospital hospital, PatientMaster patientMaster, IhiSearchCriteria ihiSearchCriteria, out string searchMessage, out searchIHIResponse ihiResponse, UserDetails user, bool checkPcehr)
        {
            bool result = false;
            X509Certificate2 tlsCert = GetMedicareCertificate(hospital);
            X509Certificate2 signingCert = tlsCert;
            ProductType product = GetProduct();
            QualifiedId userQualifiedId = GetUserQualifiedId(user);
            QualifiedId hpio = GetHpioQualifiedId(user, hospital);
            using (ConsumerSearchIHIClient client = GetTheClient(tlsCert, signingCert, product, userQualifiedId, hpio))
            {
                searchMessage = string.Empty;
                ihiResponse = new searchIHIResponse();

                // Validate the IHI - If we have one
                if (!string.IsNullOrEmpty(ihiSearchCriteria.Ihi))
                {
                    ihiSearchCriteria.MedicareNumber = null;
                    ihiSearchCriteria.MedicareIrn = null;
                    ihiSearchCriteria.DvaNumber = null;
                    int passCounter;
                    for (passCounter = 0; passCounter < ihiSearchCriteria.Names.Count; passCounter++)
                    {
                        result = SearchForIhi(hospital, patientMaster, ihiSearchCriteria, passCounter, ref searchMessage, out ihiResponse, client, result, checkPcehr, user);
                        if (ihiResponse == null || result)
                        {
                            break;
                        }
                    }

                    // After completing validation, update the PatientMasterIhi and PatientMasterIhiHistory tables
                    UpdatePatientMasterIhi(hospital, patientMaster, ihiSearchCriteria, passCounter, ihiResponse, client, checkPcehr, user);
                }

                // Search by Medicare or DVA, but only if we do not already have an IHI
                else
                {
                    ihiSearchCriteria.Ihi = null;
                    for (int passCounter = 0; passCounter < ihiSearchCriteria.Names.Count; passCounter++)
                    {
                        result = SearchForIhi(hospital, patientMaster, ihiSearchCriteria, passCounter, ref searchMessage, out ihiResponse, client, result, checkPcehr, user);
                        if (result || patientMaster.IhiStatusId == (int)IhiStatus.ServiceUnavailable)
                        {
                            // Only if the search was successful, update the PatientMasterIhi and PatientMasterIhiHistory tables
                            UpdatePatientMasterIhi(hospital, patientMaster, ihiSearchCriteria, passCounter, ihiResponse, client, checkPcehr, user);
                            break;
                        }
                    }
                }
            }
            return result;
        }

        /// <summary>
        /// Populates the the validated patient ihi object ready to be passed back to the database.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <param name="ihiResponse">The ihi response.</param>
        /// <param name="lastUpdated">The last updated.</param>
        /// <returns></returns>
        static private PatientMasterIhi PopulatePatientIhi(Hospital hospital, IhiSearchCriteria ihiSearchCriteria, searchIHIResponse ihiResponse, DateTime lastUpdated, int passCounter, string response, string request)
        {
            PatientMasterIhi result = new PatientMasterIhi();
            result.PatientMasterId = ihiSearchCriteria.PatientMasterId;

            if (passCounter == ihiSearchCriteria.Names.Count)
            {
                // If all the validations were unsuccessful, store the original
                // registered name in PatientMasterIhi with the original IHI.
                // The original registered name should be the first of the IHI
                // search criteria names, with index 0.
                // Set the IHI status to DemographicMismatch.
                passCounter = 0;
                result.Ihi = ihiSearchCriteria.Ihi;
                result.IhiRecordStatusId = (int)IhiRecordStatus.Unknown;
                result.IhiStatusId = (int)IhiStatus.DemographicMismatch;
            }
            else
            {
                result.Ihi = ihiResponse != null ? Helpers.ExtractIhi(ihiResponse.searchIHIResult.ihiNumber) : ihiSearchCriteria.Ihi;
                result.IhiRecordStatusId = ihiResponse != null ? Helpers.GetIhiRecordStatusValue(ihiResponse.searchIHIResult.ihiRecordStatus.ToString()) : (int)IhiRecordStatus.Unknown;
                result.IhiStatusId = ihiResponse != null ? Helpers.GetIhiStatusValue(ihiResponse.searchIHIResult.ihiStatus.ToString()) : (int)IhiStatus.Unknown;
                if (IsResolved(ihiResponse))
                {
                    // The PatientMasterIhiDl.RemoveUnusedIhi method will pick
                    // up this status and set the oldItem to Resolved (in history)
                    // and the newItem to Active.
                    result.IhiStatusId = (int)IhiStatus.Resolved;
                }
                if (IsRetired(ihiResponse))
                {
                    result.Ihi = ihiSearchCriteria.Ihi;
                    result.IhiStatusId = (int)IhiStatus.Retired;
                    result.IhiRecordStatusId = (int)IhiRecordStatus.Unknown;
                }
            }

            // Set the DateOfBirth with ihiSearchCriteria DateOfBirth
            if (ihiSearchCriteria.Names[passCounter].DateOfBirth.HasValue)
            {
                ihiSearchCriteria.DateOfBirth = ihiSearchCriteria.Names[passCounter].DateOfBirth.Value;
            }

            // Set the RegisteredDateOfBirth if the RegisteredDateOfBirthEnabled  to true
            if (Settings.Instance.RegisteredDateOfBirthEnabled)
            {
                if (!ihiSearchCriteria.Names[passCounter].DateOfBirth.HasValue)
                {
                    result.RegisteredDateOfBirth = ihiSearchCriteria.DateOfBirth;
                }
                else
                {
                    result.RegisteredDateOfBirth = ihiSearchCriteria.Names[passCounter].DateOfBirth;
                }
            }

            result.RegisteredFamilyName = ihiSearchCriteria.Names[passCounter].FamilyName;
            result.RegisteredGivenName = ihiSearchCriteria.Names[passCounter].GivenName;
            result.RegisteredSexId = (int)ihiSearchCriteria.Names[passCounter].Sex;
            result.DateLastValidated = lastUpdated;
            result.Response = response;
            result.Request = request;

            // Properties used to check for an existing patient whose demographics match the search criteria used.
            result.MedicareNumber = ihiSearchCriteria.MedicareNumber;
            result.MedicareNumberSequence = ihiSearchCriteria.MedicareIrn;
            result.DvaNumber = ihiSearchCriteria.DvaNumber;
            result.DateOfBirth = ihiSearchCriteria.DateOfBirth.Value;
            result.HospitalId = hospital.HospitalId.Value;
            result.HealthProviderOrganisationNetworkId = hospital.HealthProviderOrganisationNetworkId;
            
            return result;
        }

        /// <summary>
        /// Records an IHI Lookup Alert that indicates the Medicare service was unavailable when HIPS attempted
        /// to obtain the IHI for this patient. A background process will periodically attempt to obtain the IHI
        /// for patients that have this alert recorded.
        /// </summary>
        /// <param name="hospital">The hospital.</param>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="searchMessage">The message containing the exception reported by Medicare HI Service</param>
        /// <param name="client">The client containing the SOAP request / response</param>
        /// <param name="user">The user who attempted to obtain the IHI</param>
        private static void RecordServiceUnavailable(Hospital hospital, PatientMaster patientMaster, string searchMessage, ConsumerSearchIHIClient client, UserDetails user)
        {
            IhiLookupAlertBl.Insert(patientMaster.PatientMasterId.Value, searchMessage, client.SoapMessages.SoapRequest, client.SoapMessages.SoapResponse);

            PatientMasterIhi ihi = new PatientMasterIhi();
            ihi.PatientMasterId = patientMaster.PatientMasterId.Value;
            ihi.Ihi = patientMaster.Ihi;
            ihi.DateModified = DateTime.Now;
            ihi.RegisteredFamilyName = patientMaster.CurrentName.FamilyName;
            ihi.RegisteredGivenName = patientMaster.CurrentName.GivenNames;
            ihi.RegisteredSexId = patientMaster.CurrentSexId;

            // Set the status to ServiceUnavailable in both the patient master and the IHI record.
            patientMaster.IhiStatusId = ihi.IhiStatusId = (int)IhiStatus.ServiceUnavailable;
            patientMaster.IhiStatus = IhiStatus.ServiceUnavailable.ToString();
            patientMaster.IhiRecordStatusId = ihi.IhiRecordStatusId = (int)IhiRecordStatus.Unknown;
            patientMaster.IhiRecordStatus = IhiRecordStatus.Unknown.ToString();

            // Keep the original IhiLastValidated date/time to indicate that it is stale.
            // Store the SQL minimum date (1/1/1753) when the IHI has never been found.
            ihi.DateLastValidated = patientMaster.IhiLastValidated ?? SqlDateTime.MinValue.Value;

            // These properties are not stored in PatientMasterIhi but required
            // by PatientMasterIhiDl for duplicate IHI / patient checking.
            ihi.HospitalId = hospital.HospitalId.Value;
            ihi.DateOfBirth = patientMaster.DateOfBirth;
            ihi.MedicareNumber = patientMaster.MedicareNumber;
            ihi.MedicareNumberSequence = patientMaster.MedicareIrn;
            ihi.DvaNumber = patientMaster.DvaNumber;
            ihi.Request = client.SoapMessages.SoapRequest;
            ihi.Response = client.SoapMessages.SoapResponse;

            //Retain the value in registeredDateOfBirth  when RegisteredDateofBirth Enabled
            if (Settings.Instance.RegisteredDateOfBirthEnabled)
            {
                ihi.RegisteredDateOfBirth = patientMaster.RegisteredDateOfBirth;
            }

            PatientMasterIhiDl dataAccess = new PatientMasterIhiDl(user);
            dataAccess.Update(ihi);
            PendingIhiPcehrLookupDl pendingLookupDataAccess = new PendingIhiPcehrLookupDl(user);
            pendingLookupDataAccess.Save(patientMaster, hospital);
        }

        /// <summary>
        /// Searches for ihi - Single pass.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <param name="passCounter">The pass counter.</param>
        /// <param name="searchMessage">The search message.</param>
        /// <param name="ihiResponse">The ihi response.</param>
        /// <param name="client">The client.</param>
        /// <param name="result">if set to <c>true</c> [result].</param>
        /// <param name="checkPcehr">if set to <c>true</c> [check pcehr].</param>
        /// <param name="user">The user.</param>
        /// <returns></returns>
        static private bool SearchForIhi(Hospital hospital, PatientMaster patientMaster, IhiSearchCriteria ihiSearchCriteria, int passCounter, ref string searchMessage, out searchIHIResponse ihiResponse, ConsumerSearchIHIClient client, bool result, bool checkPcehr, UserDetails user)
        {
            searchIHI search = GetSearchDetails(ihiSearchCriteria, passCounter);
            searchMessage = string.Empty;
            ihiResponse = null;
            try
            {
                if (!string.IsNullOrEmpty(ihiSearchCriteria.Ihi))
                {
                    search.ihiNumber = Helpers.InsertIhiIdentifier(search.ihiNumber);
                    ihiResponse = client.BasicSearch(search);
                }
                else
                {
                    string savedMedicare = search.medicareCardNumber;
                    string savedIrn = search.medicareIRN;
                    string savedDva = search.dvaFileNumber;
                    if (!string.IsNullOrEmpty(ihiSearchCriteria.MedicareNumber)
                        && ihiSearchCriteria.IsMedicareNumberValid.HasValue
                        && ihiSearchCriteria.IsMedicareNumberValid.Value)
                    {
                        search.dvaFileNumber = null;
                        ihiResponse = client.BasicMedicareSearch(search);
                    }

                    if ((ihiResponse == null || ihiResponse.searchIHIResult == null || string.IsNullOrEmpty(ihiResponse.searchIHIResult.ihiNumber))
                        && !string.IsNullOrEmpty(savedDva)
                        && savedDva.Length > 2)
                    {
                        if (ihiResponse != null)
                        {
                            // Before we try the DVA search, make sure the Medicare search gets audited.
                            Audit(ihiResponse, ihiSearchCriteria, user, hospital, searchMessage, passCounter, client.SoapMessages.SoapResponse, client.SoapMessages.SoapRequest);
                        }
                        search.dvaFileNumber = savedDva;
                        search.medicareIRN = null;
                        search.medicareCardNumber = null;
                        ihiResponse = client.BasicDvaSearch(search);
                    }
                    search.medicareCardNumber = savedMedicare;
                    search.medicareIRN = savedIrn;
                    search.dvaFileNumber = savedDva;
                }

                if (ihiResponse == null)
                {
                    searchMessage = "Insufficient information for IHI search";
                    return false;
                }
                else if (string.IsNullOrEmpty(ihiResponse.searchIHIResult.ihiNumber))
                {
                    StringBuilder message = new StringBuilder();
                    foreach (ServiceMessageType error in ihiResponse.searchIHIResult.serviceMessages.serviceMessage)
                    {
                        message.AppendLine(string.Format("{0} - {1}", error.details, error.reason));
                    }
                    searchMessage = message.ToString();
                    IhiLookupAlertBl.Insert(ihiSearchCriteria.PatientMasterId, searchMessage, client.SoapMessages.SoapRequest, client.SoapMessages.SoapResponse);
                    return false;
                }
                else
                {
                    ihiSearchCriteria.Ihi = Helpers.ExtractIhi(ihiResponse.searchIHIResult.ihiNumber);
                }

                result = true;
            }
            catch (System.ServiceModel.FaultException<ServiceMessagesType> ex)
            {
                bool shouldRethrowException = true;
                StringBuilder message = new StringBuilder();
                if (ex.Detail != null && ex.Detail.serviceMessage != null)
                {
                    for (int i = 0; i < ex.Detail.serviceMessage.Length; i++)
                    {
                        ServiceMessageType msg = ex.Detail.serviceMessage[i];
                        if (i > 1)
                        {
                            message.Append("\n\n");
                        }
                        if (ex.Detail.serviceMessage.Length != 1)
                        {
                            message.Append(i + 1);
                            message.Append(": ");
                        }
                        message.Append(msg.reason);
                    }

                    // Something nasty happend. Need to reset the IHI
                    ihiResponse = new searchIHIResponse();
                    ihiResponse.searchIHIResult = new searchIHIResult();
                    ihiResponse.searchIHIResult.ihiNumber = string.Empty;
                    ihiResponse.searchIHIResult.serviceMessages = new ServiceMessagesType();
                    ihiResponse.searchIHIResult.serviceMessages.serviceMessage = ex.Detail.serviceMessage;
                    UpdatePatientMasterIhi(hospital, patientMaster, ihiSearchCriteria, passCounter, ihiResponse, client, checkPcehr, user);
                    if (IsRetired(ihiResponse))
                    {
                        shouldRethrowException = false;
                        result = true;
                    }
                    searchMessage = message.ToString();
                    IhiLookupAlertBl.Insert(ihiSearchCriteria.PatientMasterId, message.ToString(), client.SoapMessages.SoapRequest, client.SoapMessages.SoapResponse);
                }
                else
                {
                    message.Append("Service is not available. Please try again later: " + ex.Message);
                    searchMessage = message.ToString();
                    RecordServiceUnavailable(hospital, patientMaster, ex.Message, client, user);
                }
                if (shouldRethrowException)
                {
                    throw new IhiServiceUnavailableException(searchMessage, ex);
                }
            }
            catch (ArgumentException ex)
            {
                searchMessage = "Not all the fields have been completed. Please review and try again: " + ex.Message;
                IhiLookupAlertBl.Insert(ihiSearchCriteria.PatientMasterId, searchMessage, client.SoapMessages.SoapRequest, client.SoapMessages.SoapResponse);
                return false;
            }
            catch (Exception ex)
            {
                // The exception is caught here for cases like:
                //
                // 1. No such operation 'mcaSearchEHPConsumerRequest'
                //
                // 2. The underlying connection was closed: A connection that was expected to be kept alive was
                //    closed by the server.
                //
                // 3. There was no endpoint listening at https://www3.medicareaustralia.gov.au/pcert/soap/services/
                //    that could accept the message. This is often caused by an incorrect address or SOAP action.
                //    See InnerException, if present, for more details.
                //
                searchMessage = "IHI Lookup service is not available. Please try again later: " + ex.Message;
                RecordServiceUnavailable(hospital, patientMaster, searchMessage, client, user);

                // These exceptions must filter up to System Error Log.
                throw new IhiServiceUnavailableException(searchMessage, ex);
            }
            finally
            {
                // We need to audit but not update the patient IHI details here.
                // The update must happen when all passes are completed (for validation)
                // or as soon as there is a positive result (for search).
                Audit(ihiResponse, ihiSearchCriteria, user, hospital, searchMessage, passCounter,
                    client.SoapMessages.SoapResponse, client.SoapMessages.SoapRequest);
            }
            return result;
        }

        /// <summary>
        /// Updates the patient master ihi.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <param name="passCounter">The pass counter.</param>
        /// <param name="ihiResponse">The ihi response.</param>
        /// <param name="client">The client.</param>
        /// <param name="checkPcehr">if set to <c>true</c> [check pcehr].</param>
        /// <param name="user">The user.</param>
        private static void UpdatePatientMasterIhi(Hospital hospital, PatientMaster patientMaster, IhiSearchCriteria ihiSearchCriteria, int passCounter, searchIHIResponse ihiResponse, ConsumerSearchIHIClient client, bool checkPcehr, UserDetails user)
        {
            PatientMasterIhiDl dataAccess = new PatientMasterIhiDl(user);
            DateTime lastUpdated = DateTime.Now;
            PatientMasterIhi item;

            // If the IHI search found no matches then we need to get the last RegisteredDateofBirth to properly set the PatientMasterIHI record.
            // NOTE: When RegisteredDateOfBirthEnabled is false the Date of Birth is returned by the patientMaster.RegisteredDateOfBirth field.
            if (passCounter == ihiSearchCriteria.Names.Count)
            {
                ihiSearchCriteria.DateOfBirth = patientMaster.RegisteredDateOfBirth;
            }

            item = PopulatePatientIhi(hospital, ihiSearchCriteria, ihiResponse, lastUpdated, passCounter, client.SoapMessages.SoapResponse, client.SoapMessages.SoapRequest);

            // Calling the data access Update method will change the IhiStatusId and IhiRecordStatusId if the IHI cannot be assigned to the patient record (e.g. duplicate).
            if (!dataAccess.Update(item))
            {
                throw new Exception(string.Format(ConstantsResource.DatabaseError, dataAccess.GetType().FullName));
            }

            // Set the final values back into the patient master for further use by the application.
            patientMaster.Ihi = item.Ihi;
            patientMaster.IhiLastValidated = item.DateLastValidated;
            patientMaster.IhiRecordStatusId = item.IhiRecordStatusId;
            patientMaster.IhiStatusId = item.IhiStatusId;
            patientMaster.RegisteredFamilyName = item.RegisteredFamilyName;
            patientMaster.RegisteredGivenName = item.RegisteredGivenName;
            patientMaster.RegisteredSexId = item.RegisteredSexId;

            if (Settings.Instance.RegisteredDateOfBirthEnabled)
            {
                patientMaster.RegisteredDateOfBirth = item.RegisteredDateOfBirth.Value;
            }
            // Do the override before the PCEHR check, in case the PCEHR check fails
            UpdatePatientMasterOverride(patientMaster, ihiSearchCriteria);

            // We should only check PCEHR if we have an IHI and it is active
            if (checkPcehr && !string.IsNullOrEmpty(patientMaster.Ihi) && item.IhiStatusId == (int)IhiStatus.Active)
            {
                DoesPcehrExist doesPcehrExist = new DoesPcehrExist();
                DoesPcehrExistResponse result = doesPcehrExist.PcehrExists(null, hospital, patientMaster, user);
                if (result.HipsResponse.Status != HipsResponseIndicator.OK)
                {
                    throw new PcehrServiceUnavailableException(result.HipsResponse, null);
                }
                ihiSearchCriteria.PcehrExists = result.PcehrExists;
            }

            ihiSearchCriteria.ReturnedIhiRecordStatus = (IhiRecordStatus)item.IhiRecordStatusId;
            ihiSearchCriteria.ReturnedIhiStatus = (IhiStatus)item.IhiStatusId;
            ihiSearchCriteria.Ihi = item.UsableIhi;
            ihiSearchCriteria.IhiLastUpdated = lastUpdated;
        }

        /// <summary>
        /// Updates or deletes the patient master override.
        /// </summary>
        /// <param name="patientMaster">The patient master.</param>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        private static void UpdatePatientMasterOverride(PatientMaster patientMaster, IhiSearchCriteria ihiSearchCriteria)
        {
            if (!HIPS.Configuration.Settings.Instance.PatientMasterOverride)
            {
                return;
            }

            PatientMasterOverride patientMasterOverride = new PatientMasterOverride();
            PatientMasterOverrideDl dataAccess = new PatientMasterOverrideDl();

            patientMasterOverride.PatientMasterId = patientMaster.PatientMasterId.Value;

            // If the IHI was not found, delete the PUMA override, as it has become out of sync with the HI Service data, and we should revert to the current PAS information.
            if (string.IsNullOrEmpty(patientMaster.Ihi))
            {
                dataAccess.Delete(patientMasterOverride);
            }
            else
            {
                bool overrideRequired = false;

                //if (patientMaster.DateOfBirth != ihiSearchCriteria.DateOfBirth)
                //{
                //    patientMasterOverride.DateOfBirth = ihiSearchCriteria.DateOfBirth.Value;
                //    overrideRequired = true;
                //}

                if (!Settings.Instance.RegisteredDateOfBirthEnabled && patientMaster.DateOfBirth != ihiSearchCriteria.DateOfBirth)
                {
                    patientMasterOverride.DateOfBirth = ihiSearchCriteria.DateOfBirth.Value;
                    overrideRequired = true;
                }

                // Store a new override only if the Medicare or DVA number was used in the search. These will be null on a validation, but should not cause a new override to be stored with null.
                if (ihiSearchCriteria.MedicareNumber != null && patientMaster.MedicareNumber != ihiSearchCriteria.MedicareNumber)
                {
                    patientMasterOverride.MedicareNumber = ihiSearchCriteria.MedicareNumber;
                    patientMasterOverride.IsMedicareNumberValid = true;
                    overrideRequired = true;
                }
                if (ihiSearchCriteria.MedicareIrn != null && patientMaster.MedicareIrn != ihiSearchCriteria.MedicareIrn)
                {
                    patientMasterOverride.MedicareIrn = ihiSearchCriteria.MedicareIrn;
                    overrideRequired = true;
                }
                if (ihiSearchCriteria.DvaNumber != null && patientMaster.DvaNumber != ihiSearchCriteria.DvaNumber)
                {
                    patientMasterOverride.DvaNumber = ihiSearchCriteria.DvaNumber;
                    overrideRequired = true;
                }
                if (overrideRequired)
                {
                    patientMasterOverride.DateModified = new DateTime(2000, 1, 1);
                    dataAccess.Save(patientMasterOverride);
                }
            }
        }

        /// <summary>
        /// Validates the parameters for all names being passed in.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <returns>whether valid</returns>
        static private bool ValidateSearchCriteria(IhiSearchCriteria ihiSearchCriteria)
        {
            bool result = true;
            if (string.IsNullOrEmpty(ihiSearchCriteria.MedicareNumber) && string.IsNullOrEmpty(ihiSearchCriteria.DvaNumber) && string.IsNullOrEmpty(ihiSearchCriteria.Ihi))
            {
                return false;
            }

            if (!ihiSearchCriteria.DateOfBirth.HasValue)
            {
                return false;
            }

            foreach (IhiSearchCriteriaName name in ihiSearchCriteria.Names)
            {
                result = ValidateSearchCriteriaName(name);
                if (!result)
                {
                    break;
                }
            }
            return result;
        }

        /// <summary>
        /// Validates the parameters.
        /// </summary>
        /// <param name="ihiSearchCriteria">The ihi search criteria.</param>
        /// <returns></returns>
        static private bool ValidateSearchCriteriaName(IhiSearchCriteriaName ihiSearchCriteriaName)
        {
            bool result = true;

            if (string.IsNullOrEmpty(ihiSearchCriteriaName.FamilyName) || string.IsNullOrEmpty(ihiSearchCriteriaName.GivenName) || ihiSearchCriteriaName.Sex == SexEnumerator.NotStatedOrInadequatelyDescribed)
            {
                result = false;
            }
            return result;
        }
    }
}